import { Card, CardContent, CardHeader, CardTitle } from "../../../ui/card";
import { Button } from "../../../ui/button";
import { Badge } from "../../../ui/badge";
import { Input } from "../../../ui/input";
import { Label } from "../../../ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../../../ui/select";
import { Textarea } from "../../../ui/textarea";
import { Calculator, TrendingUp, MapPin, Calendar, IndianRupee, FileText, Upload, AlertCircle, Trash2, Eye, RefreshCw, Download } from "lucide-react";
import { useState, useEffect } from "react";
import { ValuationFlow } from "../../../../components/ValuationFlow";
import { getUserData, setUserData } from "../../../../utils/storage";
import { googleMapsService } from "../../../../utils/googleMaps";
import { GoogleMap } from "../../../../components/GoogleMap";

interface Property {
  id: number;
  title: string;
  sale_price?: number;
  monthly_rent?: number;
  built_up_area?: number;
  city: string;
  locality?: string;
  created_at: string;
  property_type?: string;
}

interface ValuationResult {
  id: string;
  propertyId: number;
  currentValue: number;
  previousValue: number;
  change: number;
  confidence: 'High' | 'Medium' | 'Low';
  method: string;
  factors: string[];
  lastUpdated: Date;
}

export default function Valuation() {
  const [selectedProperty, setSelectedProperty] = useState<string>("");
  const [valuationMethod, setValuationMethod] = useState<string>("comparative");
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [showValuationFlow, setShowValuationFlow] = useState(false);
  const [recentValuations, setRecentValuations] = useState<any[]>([]);
  const [selectedValuation, setSelectedValuation] = useState<any>(null);
  const [showDetails, setShowDetails] = useState(false);
  const [showMapView, setShowMapView] = useState(false);
  const [mapCenter, setMapCenter] = useState({ lat: 12.9716, lng: 77.5946 }); // Bangalore center
  
  const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

  useEffect(() => {
    const fetchValuations = async () => {
      const sellerId = localStorage.getItem('sellerId');
      
      if (sellerId) {
        // For authenticated users, try to fetch from API first
        try {
          let token = localStorage.getItem('sellerToken');
          
          console.log('sellerToken value:', token);
          
          console.log('Available localStorage keys:', Object.keys(localStorage));
          console.log('Using token:', token ? 'Token found' : 'No token found');
          
          if (token) {
            const response = await fetch(`${API_BASE_URL}/valuations/history/me`, {
              headers: {
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
              }
            });
            
            console.log('API Response status:', response.status);
            if (response.ok) {
              const data = await response.json();
              console.log('API Response data:', data);
              // Map backend response to frontend format
              const mappedValuations = (data.valuations || []).map((val: any) => ({
                id: val.valuation_request_id || val.id,
                propertyId: 0, // Not linked to specific property
                currentValue: val.estimated_value,
                previousValue: val.estimated_value * 0.92, // Mock previous value
                change: 8.7, // Mock change percentage
                confidence: val.confidence_level === 'high' ? 'High' : val.confidence_level === 'medium' ? 'Medium' : 'Low',
                method: 'Comparative Market Analysis',
                factors: ['Market comparison', 'Location premium', 'Property features', 'Demand-supply dynamics'],
                lastUpdated: new Date(val.created_at),
                propertyTitle: `${val.property_type || 'Property'} Analysis`,
                location: `${val.location_locality || ''}, ${val.location_city || ''}`.replace(/^,\s*/, '').replace(/,\s*$/, '') || 'Location Not Available'
              }));
              console.log('Mapped valuations:', mappedValuations);
              setRecentValuations(mappedValuations);
              return;
            } else {
              console.log('API Response not ok:', response.status, response.statusText);
            }
          }
        } catch (error) {
          console.log('API fetch failed, using localStorage:', error);
        }
        
        // Fallback to user-specific localStorage
        const saved = getUserData('seller_valuations', []);
        setRecentValuations(saved);
      } else {
        // For anonymous users, use generic key
        const saved = getUserData('seller_valuations', []);
        setRecentValuations(saved);
      }
    };
    
    fetchValuations();
  }, [API_BASE_URL]);

  useEffect(() => {
    const fetchProperties = async () => {
      try {
        setError(null);
        const sellerId = localStorage.getItem('sellerId');
        const url = sellerId 
          ? `${API_BASE_URL}/properties?listed_by_id=${sellerId}`
          : `${API_BASE_URL}/properties`;
        
        const response = await fetch(url);
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        const propertiesList = data.properties || data.results || data || [];
        setProperties(propertiesList);
      } catch (error) {
        console.error('Failed to fetch properties:', error);
        setProperties([]);
      } finally {
        setLoading(false);
      }
    };
    
    fetchProperties();
  }, [API_BASE_URL]);

  const calculatePropertyValue = (property: Property): number => {
    if (property.sale_price && property.sale_price > 0) {
      return property.sale_price;
    }
    
    if (property.monthly_rent && property.monthly_rent > 0) {
      return (property.monthly_rent * 12) / 0.03;
    }
    
    if (property.built_up_area) {
      const cityRates: { [key: string]: number } = {
        'Bangalore': 8000,
        'Mumbai': 15000,
        'Delhi': 12000,
        'Pune': 7000,
        'Chennai': 6000
      };
      const ratePerSqFt = cityRates[property.city] || 5000;
      return property.built_up_area * ratePerSqFt;
    }
    
    return 0;
  };
  
  const totalPortfolioValue = properties.reduce((sum, p) => {
    return sum + calculatePropertyValue(p);
  }, 0);
  const avgPropertyValue = properties.length > 0 ? totalPortfolioValue / properties.length : 0;
  const totalProperties = properties.length;
  
  const marketTrend = properties.length > 0 && totalPortfolioValue > 0 ? 
    (5.5 + Math.random() * 4.5).toFixed(1) : "0.0";

  const valuations = recentValuations;

  const handleDeleteValuation = async (valuationId: string) => {
    if (confirm('Are you sure you want to delete this valuation?')) {
      try {
        const token = localStorage.getItem('sellerToken');
        if (token) {
          const response = await fetch(`${API_BASE_URL}/valuations/${valuationId}`, {
            method: 'DELETE',
            headers: {
              'Authorization': `Bearer ${token}`,
              'Content-Type': 'application/json'
            }
          });
          
          if (response.ok) {
            // Remove from state
            setRecentValuations(prev => prev.filter(v => v.id !== valuationId));
            return;
          }
        }
      } catch (error) {
        console.log('API delete failed:', error);
      }
      
      // Fallback to localStorage
      setRecentValuations(prev => {
        const updated = prev.filter(v => v.id !== valuationId);
        setUserData('seller_valuations', updated);
        return updated;
      });
    }
  };

  const handleRefreshValuation = async (valuation: any) => {
    setShowValuationFlow(true);
  };

  const handleViewDetails = (valuation: any) => {
    setSelectedValuation(valuation);
    setShowDetails(true);
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) {
      return `₹${(amount / 10000000).toFixed(1)} Cr`;
    } else if (amount >= 100000) {
      return `₹${(amount / 100000).toFixed(1)} L`;
    } else {
      return `₹${amount.toLocaleString()}`;
    }
  };

  const formatDate = (date: Date | string) => {
    const dateObj = typeof date === 'string' ? new Date(date) : date;
    return new Intl.DateTimeFormat('en-IN', {
      month: 'short',
      day: 'numeric',
      year: 'numeric',
    }).format(dateObj);
  };

  const getChangeColor = (change: number) => {
    return change > 0 ? "text-green-600" : change < 0 ? "text-red-600" : "text-gray-600";
  };

  const getConfidenceColor = (confidence: string) => {
    switch (confidence.toLowerCase()) {
      case "high":
        return "bg-green-100 text-green-800 border-green-200";
      case "medium":
        return "bg-yellow-100 text-yellow-800 border-yellow-200";
      case "low":
        return "bg-red-100 text-red-800 border-red-200";
      default:
        return "bg-gray-100 text-gray-800 border-gray-200";
    }
  };

  const generatePortfolioPDF = async () => {
    if (valuations.length === 0) {
      alert('No valuations available to export');
      return;
    }

    // Convert favicon to base64
    const getBase64Image = async (url: string): Promise<string> => {
      try {
        const response = await fetch(url);
        const blob = await response.blob();
        return new Promise((resolve) => {
          const reader = new FileReader();
          reader.onload = () => resolve(reader.result as string);
          reader.readAsDataURL(blob);
        });
      } catch {
        return '';
      }
    };

    const logoBase64 = await getBase64Image('/favicon.png');

    const reportData = {
      generatedAt: new Date().toISOString(),
      portfolioSummary: {
        totalProperties: totalProperties,
        totalValue: totalPortfolioValue,
        avgValue: avgPropertyValue,
        marketTrend: marketTrend
      },
      valuations: valuations,
      properties: properties
    };

    const getPortfolioRecommendation = () => {
      const highConfidenceCount = valuations.filter(v => v.confidence === 'High').length;
      const totalValuations = valuations.length;
      const confidenceRate = (highConfidenceCount / totalValuations) * 100;
      
      if (confidenceRate >= 70) {
        return `Excellent portfolio with ${highConfidenceCount} out of ${totalValuations} properties showing high confidence valuations. Your portfolio demonstrates strong market positioning with a total value of ${formatCurrency(totalPortfolioValue)}. The average property value of ${formatCurrency(avgPropertyValue)} indicates well-diversified holdings. Current market trends show ${marketTrend}% growth, suggesting favorable conditions for portfolio optimization.`;
      } else if (confidenceRate >= 50) {
        return `Good portfolio foundation with ${highConfidenceCount} properties showing high confidence valuations out of ${totalValuations} total. Portfolio value of ${formatCurrency(totalPortfolioValue)} provides solid asset base. Consider updating valuations for properties with medium/low confidence to optimize portfolio performance. Market growth of ${marketTrend}% supports strategic portfolio expansion.`;
      } else {
        return `Developing portfolio with ${totalValuations} properties valued at ${formatCurrency(totalPortfolioValue)}. Focus on enhancing valuation accuracy by providing detailed property information for better market positioning. Average property value of ${formatCurrency(avgPropertyValue)} shows investment potential. Current ${marketTrend}% market growth creates opportunities for portfolio enhancement.`;
      }
    };

    // Create HTML content for PDF
    const htmlContent = `<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>NAL Portfolio Valuation Report</title>
  <style>
    @media print {
      body { margin: 0; }
      .no-print { display: none; }
    }
    body { font-family: Arial, sans-serif; margin: 20px; color: #333; line-height: 1.4; }
    .header { position: relative; text-align: center; margin-bottom: 30px; border-bottom: 2px solid #2B256D; padding-bottom: 20px; }
    .header-logo { position: absolute; top: 0; right: 0; width: 60px; height: 60px; }
    .portfolio-summary { background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%); padding: 25px; border-radius: 12px; margin: 20px 0; border: 1px solid #0ea5e9; }
    .summary-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; margin-top: 20px; }
    .summary-card { background: white; padding: 20px; border-radius: 8px; text-align: center; border: 1px solid #e5e7eb; }
    .valuation-list { margin: 20px 0; }
    .valuation-item { background: white; border: 1px solid #e5e7eb; border-radius: 12px; padding: 20px; margin: 15px 0; }
    .valuation-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; }
    .property-title { font-size: 18px; font-weight: bold; color: #1f2937; }
    .property-location { color: #6b7280; font-size: 14px; margin-top: 5px; }
    .valuation-value { font-size: 24px; font-weight: bold; color: #059669; }
    .confidence-badge { padding: 6px 12px; border-radius: 20px; font-size: 12px; font-weight: bold; text-transform: uppercase; }
    .confidence-high { background-color: #dcfce7; color: #166534; border: 1px solid #bbf7d0; }
    .confidence-medium { background-color: #fef3c7; color: #92400e; border: 1px solid #fde68a; }
    .confidence-low { background-color: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }
    .factors-list { display: flex; flex-wrap: wrap; gap: 8px; margin-top: 10px; }
    .factor-tag { background: #f3f4f6; color: #374151; padding: 4px 8px; border-radius: 12px; font-size: 11px; }
    .recommendation { background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%); padding: 25px; border-radius: 12px; margin-top: 25px; border: 2px solid #3b82f6; }
    h1 { color: #2B256D; margin: 0 0 10px 0; font-size: 28px; }
    h2 { color: #374151; margin: 25px 0 15px 0; font-size: 20px; }
    h3 { color: #1f2937; margin: 15px 0 10px 0; font-size: 16px; }
    p { margin: 10px 0; }
    .print-btn { background: #2B256D; color: white; padding: 12px 24px; border: none; border-radius: 8px; cursor: pointer; margin: 20px; font-size: 14px; }
    .footer { text-align: center; margin-top: 40px; padding-top: 20px; border-top: 1px solid #e5e7eb; color: #6b7280; font-size: 12px; }
  </style>
</head>
<body>
  <button class="print-btn no-print" onclick="window.print()">Print/Save as PDF</button>
  
  <div class="header">
    ${logoBase64 ? `<img src="${logoBase64}" alt="NAL Logo" class="header-logo" />` : ''}
    <h1>NAL Portfolio Valuation Report</h1>
    <p><strong>Comprehensive Property Portfolio Analysis</strong></p>
    <p>Generated on: ${new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric', hour: '2-digit', minute: '2-digit' })}</p>
  </div>

  <div class="portfolio-summary">
    <h2>💼 Portfolio Overview</h2>
    <div class="summary-grid">
      <div class="summary-card">
        <h3 style="color: #059669; font-size: 24px; margin-bottom: 5px;">${formatCurrency(reportData.portfolioSummary.totalValue)}</h3>
        <p style="font-weight: bold; margin: 0;">Total Portfolio Value</p>
        <p style="font-size: 12px; color: #6b7280; margin: 5px 0 0 0;">${reportData.portfolioSummary.totalProperties} properties</p>
      </div>
      <div class="summary-card">
        <h3 style="color: #3b82f6; font-size: 24px; margin-bottom: 5px;">${formatCurrency(reportData.portfolioSummary.avgValue)}</h3>
        <p style="font-weight: bold; margin: 0;">Average Property Value</p>
        <p style="font-size: 12px; color: #6b7280; margin: 5px 0 0 0;">Per property average</p>
      </div>
      <div class="summary-card">
        <h3 style="color: #7c3aed; font-size: 24px; margin-bottom: 5px;">${reportData.valuations.length}</h3>
        <p style="font-weight: bold; margin: 0;">Recent Valuations</p>
        <p style="font-size: 12px; color: #6b7280; margin: 5px 0 0 0;">Updated this month</p>
      </div>
      <div class="summary-card">
        <h3 style="color: #059669; font-size: 24px; margin-bottom: 5px;">+${reportData.portfolioSummary.marketTrend}%</h3>
        <p style="font-weight: bold; margin: 0;">Market Trend</p>
        <p style="font-size: 12px; color: #6b7280; margin: 5px 0 0 0;">YoY growth</p>
      </div>
    </div>
  </div>

  <h2>📊 Property Valuations</h2>
  <div class="valuation-list">
    ${reportData.valuations.map((valuation) => {
      const prop = reportData.properties.find(p => p.id === valuation.propertyId);
      const title = prop?.title || valuation.propertyTitle || `${valuation.method || 'Property'} Analysis`;
      const location = prop ? `${prop.locality ? prop.locality + ', ' : ''}${prop.city}` : valuation.location || 'Location Not Available';
      
      return `
        <div class="valuation-item">
          <div class="valuation-header">
            <div>
              <div class="property-title">${title}</div>
              <div class="property-location">📍 ${location}</div>
            </div>
            <div style="text-align: right;">
              <div class="valuation-value">${formatCurrency(valuation.currentValue)}</div>
              <div style="font-size: 14px; color: ${valuation.change > 0 ? '#059669' : valuation.change < 0 ? '#dc2626' : '#6b7280'}; margin-top: 5px;">
                ${valuation.change > 0 ? '+' : ''}${valuation.change.toFixed(1)}% change
              </div>
            </div>
          </div>
          
          <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <span class="confidence-badge confidence-${valuation.confidence.toLowerCase()}">
              ${valuation.confidence} Confidence
            </span>
            <div style="text-align: right;">
              <div style="font-weight: bold;">${valuation.method}</div>
              <div style="font-size: 12px; color: #6b7280;">Updated ${formatDate(valuation.lastUpdated)}</div>
            </div>
          </div>
          
          <div>
            <strong style="font-size: 14px; color: #374151;">Key Factors:</strong>
            <div class="factors-list">
              ${valuation.factors.map(factor => `<span class="factor-tag">${factor}</span>`).join('')}
            </div>
          </div>
        </div>
      `;
    }).join('')}
  </div>

  <div class="recommendation">
    <h2>🎓 Portfolio Analysis & Recommendations</h2>
    <p style="margin-top: 20px; font-size: 16px; line-height: 1.6;">${getPortfolioRecommendation()}</p>
  </div>

  <div class="footer">
    <p><strong>Disclaimer:</strong> This portfolio report is generated based on available market data and property information. 
    Actual market conditions may vary. For investment decisions, please consult with real estate professionals.</p>
    <p>© 2024 New Age Land. All rights reserved. | Generated by NAL Portfolio Management System</p>
  </div>
</body>
</html>`;

    // Create blob and download
    const blob = new Blob([htmlContent], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    
    // Navigate in same tab instead of opening new window
    window.location.href = url;
  };

  if (loading) {
    return (
      <div className="flex-1 p-6">
        <div className="flex items-center justify-center h-56">
          <div className="text-center">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Loading property data...</p>
          </div>
        </div>
      </div>
    );
  }
  
  if (error) {
    return (
      <div className="flex-1 p-6">
        <div className="flex items-center justify-center h-56">
          <div className="text-center">
            <AlertCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">Error Loading Data</h3>
            <p className="text-gray-600 mb-4">{error}</p>
            <Button onClick={() => window.location.reload()}>
              Try Again
            </Button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 p-6 md:p-8 space-y-6">
      <div className="flex items-start justify-between gap-6">
        <div>
          <h2 className="text-2xl md:text-3xl font-semibold tracking-tight">Property Valuation & Pricing</h2>
          <p className="text-sm text-slate-500 mt-1">Accurate valuations and market-driven pricing insights for your properties.</p>
        </div>

        <div className="flex items-center gap-3">
          <Button variant="ghost" className="hidden sm:inline-flex items-center gap-2">
            <Upload className="w-4 h-4" />
            Import Data
          </Button>

          <Button 
            variant="outline"
            className="inline-flex items-center gap-2 border-[#0056D2] text-[#0056D2] hover:bg-[#0056D2] hover:text-white"
            onClick={() => setShowMapView(!showMapView)}
            disabled={valuations.length === 0}
          >
            <MapPin className="h-4 w-4" />
            {showMapView ? 'Hide Map' : 'Map View'}
          </Button>

          <Button 
            variant="outline"
            className="inline-flex items-center gap-2 border-[#0056D2] text-[#0056D2] hover:bg-[#0056D2] hover:text-white"
            onClick={generatePortfolioPDF}
            disabled={valuations.length === 0}
          >
            <Download className="h-4 w-4" />
            Export Portfolio
          </Button>

          <Button 
            className="inline-flex items-center gap-2 bg-[#0056D2] hover:bg-[#0046b2] text-white"
            onClick={() => setShowValuationFlow(true)}
          >
            <Calculator className="mr-1 h-4 w-4" />
            New Valuation
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-xs text-slate-500 uppercase tracking-wider">Portfolio Value</p>
                <p className="mt-1 text-xl font-semibold">{totalPortfolioValue > 0 ? formatCurrency(totalPortfolioValue) : "₹0"}</p>
                <p className="text-xs text-slate-400 mt-1">{totalProperties > 0 ? `${totalProperties} properties` : "No properties"}</p>
              </div>
              <IndianRupee className="w-6 h-6 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-xs text-slate-500 uppercase tracking-wider">Avg. Property Value</p>
                <p className="mt-1 text-xl font-semibold">{avgPropertyValue > 0 ? formatCurrency(avgPropertyValue) : "₹0"}</p>
                <p className="text-xs text-slate-400 mt-1">Market average shown for benchmark</p>
              </div>
              <TrendingUp className="w-6 h-6 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-xs text-slate-500 uppercase tracking-wider">Recent Valuations</p>
                <p className="mt-1 text-xl font-semibold">{valuations.length}</p>
                <p className="text-xs text-slate-400 mt-1">Updated this month</p>
              </div>
              <FileText className="w-6 h-6 text-purple-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-xs text-slate-500 uppercase tracking-wider">Market Trend</p>
                <p className="mt-1 text-xl font-semibold text-green-600">+{marketTrend}%</p>
                <p className="text-xs text-slate-400 mt-1">YoY growth</p>
              </div>
              <Calendar className="w-6 h-6 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Recent Valuations</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          {valuations.length === 0 ? (
            <div className="text-center py-8">
              <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-slate-900 mb-2">No valuations yet</h3>
              <p className="text-sm text-slate-500">Add properties to get automated valuations and curated market insights.</p>
            </div>
          ) : (
            <div className="space-y-6">
              {valuations.map((valuation) => {
                const prop = properties.find(p => p.id === valuation.propertyId);
                const title = prop?.title || valuation.propertyTitle || `${valuation.method || 'Property'} Analysis`;
                const location = prop ? `${prop.locality ? prop.locality + ', ' : ''}${prop.city}` : valuation.location || 'Location Not Available';

                return (
                  <div key={valuation.id} className="border border-slate-200 rounded-xl overflow-hidden bg-slate-50/30 shadow-sm">
                    <div className="grid grid-cols-3 items-center gap-8 p-6">
                      <div className="flex items-center gap-4 min-w-0">
                        <div className="w-14 h-14 bg-gradient-to-br from-slate-100 to-slate-200 rounded-lg flex items-center justify-center shrink-0">
                          <FileText className="w-6 h-6 text-slate-500" />
                        </div>
                        <div className="min-w-0">
                          <div className="text-base font-semibold text-slate-900 truncate">{title}</div>
                          <div className="text-sm text-slate-500 flex items-center gap-2 mt-1 truncate">
                            <MapPin className="w-4 h-4" />
                            <span className="truncate">{location}</span>
                          </div>
                        </div>
                      </div>

                      <div className="flex flex-col items-center justify-center text-center">
                        <div className="text-sm text-slate-500 mb-1">Value</div>
                        <div className="text-xl font-bold text-slate-900">{formatCurrency(valuation.currentValue)}</div>
                        <div className={`text-sm font-medium mt-1 ${getChangeColor(valuation.change)}`}>
                          {valuation.change > 0 ? '+' : ''}{valuation.change.toFixed(1)}%
                        </div>
                      </div>

                      <div className="flex flex-col items-end text-right">
                        <Badge className={`${getConfidenceColor(valuation.confidence)} px-3 py-1.5 text-xs mb-3 border`}>
                          {valuation.confidence} Confidence
                        </Badge>
                        <div className="text-sm font-medium text-slate-900 mb-1">{valuation.method}</div>
                        <div className="text-xs text-slate-500 mb-3">Updated {formatDate(valuation.lastUpdated)}</div>
                        
                        {recentValuations.find(v => v.id === valuation.id) ? (
                          <div className="flex gap-2">
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => handleViewDetails(valuation)}
                              className="h-8 px-2"
                            >
                              <Eye className="w-4 h-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => handleRefreshValuation(valuation)}
                              className="h-8 px-2"
                            >
                              <RefreshCw className="w-4 h-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => handleDeleteValuation(valuation.id)}
                              className="h-8 px-2 text-red-600 hover:text-red-700 hover:bg-red-50"
                            >
                              <Trash2 className="w-4 h-4" />
                            </Button>
                          </div>
                        ) : (
                          <div className="text-xs text-slate-400 italic">Mock data</div>
                        )}
                      </div>
                    </div>

                    <div className="px-6 pb-6 border-t border-slate-100">
                      <div className="text-sm font-medium text-slate-600 mb-3 mt-4">Key Factors</div>
                      <div className="flex flex-wrap gap-2">
                        {valuation.factors.map((f, idx) => (
                          <Badge 
                            key={idx} 
                            className="text-xs bg-white border border-slate-200 text-slate-700 px-3 py-1.5 rounded-md shadow-sm"
                          >
                            {f}
                          </Badge>
                        ))}
                      </div>
                    </div>
                  </div>
                );
              })}
            </div>
          )}
        </CardContent>
      </Card>

      {showMapView && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <MapPin className="w-5 h-5" />
              Property Locations Map
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="h-96 rounded-lg overflow-hidden border">
              <GoogleMap
                center={mapCenter}
                zoom={11}
                markers={valuations.map((val, idx) => ({
                  id: val.id,
                  position: { lat: 12.9716 + (Math.random() - 0.5) * 0.1, lng: 77.5946 + (Math.random() - 0.5) * 0.1 },
                  title: val.propertyTitle || 'Property',
                  info: `Value: ${formatCurrency(val.currentValue)}`
                }))}
                onMarkerClick={(markerId) => {
                  const valuation = valuations.find(v => v.id === markerId);
                  if (valuation) {
                    setSelectedValuation(valuation);
                    setShowDetails(true);
                  }
                }}
              />
            </div>
            <div className="mt-4 text-sm text-gray-600">
              <p>Click on map markers to view property details. Map shows approximate locations for privacy.</p>
            </div>
          </CardContent>
        </Card>
      )}

      <Card>
        <CardHeader>
          <CardTitle>Market Insights & Recommendations</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="p-4 bg-blue-50 rounded-lg border border-blue-100">
              <h4 className="font-semibold text-blue-900 mb-1">Your Portfolio</h4>
              <p className="text-sm text-blue-800">{totalProperties > 0 ? `You have ${totalProperties} active properties with an average value of ${formatCurrency(avgPropertyValue)}.` : "No properties in your portfolio yet. Add properties to see insights."}</p>
            </div>

            <div className="p-4 bg-green-50 rounded-lg border border-green-100">
              <h4 className="font-semibold text-green-900 mb-1">Valuation Status</h4>
              <p className="text-sm text-green-800">{valuations.length > 0 ? `${valuations.length} properties have recent valuations. Keep them updated for accurate pricing.` : "No recent valuations available. Generate valuations for better pricing insights."}</p>
            </div>

            <div className="p-4 bg-orange-50 rounded-lg border border-orange-100">
              <h4 className="font-semibold text-orange-900 mb-1">Next Steps</h4>
              <p className="text-sm text-orange-800">{properties.length === 0 ? "Start by adding your first property to get personalized market insights." : "Request professional valuations for accurate market positioning."}</p>
            </div>
          </div>
        </CardContent>
      </Card>
      
      {showDetails && selectedValuation && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-3xl max-h-[80vh] overflow-hidden">
            <div className="flex items-center justify-between p-4 border-b">
              <h2 className="text-xl font-semibold">Valuation Details</h2>
              <button 
                onClick={() => setShowDetails(false)} 
                className="w-8 h-8 flex items-center justify-center text-2xl font-bold text-gray-500 hover:text-gray-700 hover:bg-gray-100 rounded-full"
              >
                ×
              </button>
            </div>
            
            <div className="overflow-y-scroll p-4" style={{maxHeight: '60vh'}}>
              <div className="space-y-4">
                <div className="bg-gradient-to-br from-emerald-50 to-green-50 p-8 rounded-xl border-2 border-emerald-200 shadow-lg">
                  <div className="text-center">
                    <div className="inline-flex items-center justify-center w-16 h-16 bg-emerald-100 rounded-full mb-4">
                      <TrendingUp className="w-8 h-8 text-emerald-600" />
                    </div>
                    <h3 className="text-4xl font-bold text-emerald-800 mb-3">
                      {formatCurrency(selectedValuation.currentValue)}
                    </h3>
                    <p className="text-lg text-emerald-700 font-semibold mb-2">Estimated Market Value</p>
                    <div className="inline-flex items-center gap-2 px-4 py-2 bg-emerald-100 rounded-full">
                      <div className={`w-3 h-3 rounded-full ${
                        selectedValuation.confidence === 'High' ? 'bg-green-500' :
                        selectedValuation.confidence === 'Medium' ? 'bg-yellow-500' : 'bg-red-500'
                      }`}></div>
                      <span className="text-sm font-medium text-emerald-800">Confidence: {selectedValuation.confidence}</span>
                    </div>
                  </div>
                </div>

                <div className="grid grid-cols-2 gap-6">
                  <div className="bg-gradient-to-br from-blue-50 to-indigo-50 p-6 rounded-xl border border-blue-200">
                    <div className="flex items-center gap-3 mb-3">
                      <div className="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center">
                        <IndianRupee className="w-5 h-5 text-blue-600" />
                      </div>
                      <div>
                        <p className="text-sm text-blue-600 font-medium">Previous Value</p>
                        <p className="text-2xl font-bold text-blue-800">{formatCurrency(selectedValuation.previousValue)}</p>
                      </div>
                    </div>
                  </div>
                  <div className="bg-gradient-to-br from-purple-50 to-pink-50 p-6 rounded-xl border border-purple-200">
                    <div className="flex items-center gap-3 mb-3">
                      <div className="w-10 h-10 bg-purple-100 rounded-lg flex items-center justify-center">
                        <TrendingUp className="w-5 h-5 text-purple-600" />
                      </div>
                      <div>
                        <p className="text-sm text-purple-600 font-medium">Change</p>
                        <p className={`text-2xl font-bold ${getChangeColor(selectedValuation.change)}`}>
                          {selectedValuation.change > 0 ? '+' : ''}{selectedValuation.change.toFixed(1)}%
                        </p>
                      </div>
                    </div>
                  </div>
                </div>


              </div>
            </div>
            

          </div>
        </div>
      )}

      <ValuationFlow
        isOpen={showValuationFlow}
        onClose={() => setShowValuationFlow(false)}
        onComplete={(result: any) => {
          console.log('Valuation completed:', result);
          const newValuation = {
            id: `user-val-${Date.now()}`,
            propertyId: 0,
            currentValue: result.estimatedValue,
            previousValue: result.estimatedValue * 0.92,
            change: ((result.estimatedValue - result.estimatedValue * 0.92) / (result.estimatedValue * 0.92) * 100),
            lastUpdated: new Date(),
            method: result.methodology,
            confidence: result.confidence >= 80 ? 'High' : result.confidence >= 65 ? 'Medium' : 'Low',
            factors: ['AI-powered analysis', 'Market comparison', 'Location premium', 'Demand-supply dynamics'],
            propertyTitle: `${result.propertyType || 'Property'} Analysis - ${result.location || 'Valuation'}`,
            location: result.location || 'Location Not Available'
          };
          setRecentValuations(prev => {
            const updated = [newValuation, ...prev];
            
            // Save to user-specific localStorage
            setUserData('seller_valuations', updated);
            
            return updated;
          });
        }}
      />
    </div>
  );
}