const API_BASE = (import.meta as any).env?.VITE_API_BASE || 'http://localhost:8090/api';

// Types
interface Property {
  id: string;
  title: string;
  type: string;
  bhk?: string;
  address: string;
  city: string;
  state: string;
  pincode: string;
  landmark: string;
  area: number;
  price: number;
  listingIntent: string;
  listing_type?: string;
  status: string;
  isAsset?: boolean; // Asset Management field
  listing_status?: string; // Asset Management field
  amenities: string[];
  customAmenities: string[];
  virtualTour?: string;
  sellerType: string;
  biddingEnabled: boolean;
  coordinates?: { lat: number; lng: number };
  views: number;
  inquiries: number;
  createdAt: string;
  updatedAt: string;
  description: string;
  images: Array<{ url: string; category?: string }>;
  videos: Array<{ url: string; thumbnail_url?: string; title?: string }>;
  imageCategories: { [key: string]: Array<{ url: string }> };
  bathrooms?: number;
  facing?: string;
  furnishing?: string;
  propertyAge?: string;
  possessionStatus?: string;
  nearbyPlaces?: string[];
  propertyHighlights?: string[];
  // Debug properties
  _debug_isasset?: any;
  _debug_isAsset?: any;
  _debug_rawRow?: any;
}

interface PropertyFormData {
  title: string;
  description?: string;
  type: string;
  listingIntent: string;
  address: string;
  city: string;
  state?: string;
  pincode: string;
  landmark?: string;
  area: number;
  bhk?: string;
  price: number;
  amenities: string[];
}

// Helper function to map backend data to frontend format
function mapRowToProperty(row: any): Property {
  console.log('🔍 Raw property row:', row);
  console.log('🔍 isasset field variations:', {
    isasset: row.isasset,
    isAsset: row.isAsset,
    is_asset: row.is_asset,
    typeof_isasset: typeof row.isasset
  });
  // Handle images
  let images = [];
  if (row.images && Array.isArray(row.images) && row.images.length > 0) {
    images = row.images.map((img: any) => ({
      url: img.image || img.url || img,
      category: img.category || 'interior'
    }));
  } else if (row.primary_image) {
    images = [row.primary_image];
  }
  
  // Handle videos
  let videos = [];
  if (row.videos && Array.isArray(row.videos) && row.videos.length > 0) {
    videos = row.videos.map((video: any) => ({
      url: video.url || video.video || video,
      thumbnail_url: video.thumbnail_url,
      title: video.title
    }));
  }

  return {
    id: String(row.id),
    title: row.title,
    type: row.property_type,
    bhk: row.bedrooms ? `${row.bedrooms}BHK` : undefined,
    address: row.address,
    city: row.city,
    state: row.state || 'Karnataka',
    pincode: row.pincode,
    landmark: row.landmark || row.locality || '',
    area: Number(row.built_up_area || row.carpet_area || row.super_area || 0),
    price: Number(row.sale_price || row.monthly_rent || 0),
    listingIntent: row.listing_type,
    listing_type: row.listing_type, // Asset Management field
    status: row.status || 'active',
    isAsset: Boolean(row.isasset || row.isAsset || row.is_asset || row.isasset === 1 || row.isAsset === 1 || row.isasset === true), // Asset Management field - check all variations including numeric 1
    listing_status: row.listing_status || 'listed', // Asset Management field
    amenities: Array.isArray(row.amenities) ? row.amenities : [],
    customAmenities: [],
    virtualTour: row.virtual_tour_url,
    sellerType: 'owner',
    biddingEnabled: Boolean(row.bidding_enabled),
    coordinates: (row.latitude && row.longitude) ? {
      lat: parseFloat(row.latitude),
      lng: parseFloat(row.longitude)
    } : undefined,
    views: Number(row.views || 0),
    inquiries: Number(row.inquiries || 0),
    createdAt: row.created_at,
    updatedAt: row.updated_at || row.created_at,
    description: row.description || '',
    images: images,
    videos: videos,
    imageCategories: {},
    // Additional fields for property details
    bathrooms: row.bathrooms,
    facing: row.facing,
    furnishing: row.furnishing,
    propertyAge: row.property_age,
    possessionStatus: row.possession_status,
    nearbyPlaces: Array.isArray(row.nearby_places) ? row.nearby_places : [],
    propertyHighlights: Array.isArray(row.property_highlights) ? row.property_highlights : [],
    // DEBUG: Add raw backend data to see what we're getting
    _debug_isasset: row.isasset,
    _debug_isAsset: row.isAsset,
    _debug_rawRow: row
  };
}

export const api = {
  // Create property with files
  createProperty: async (data: any): Promise<{ id: number }> => {
    // Validate required fields
    if (!data.title) throw new Error('Title is required');
    if (!data.address) throw new Error('Address is required');
    if (!data.city) throw new Error('City is required');
    
    // Get seller token if available
    const sellerToken = localStorage.getItem('sellerToken');
    
    // For asset creation (minimal form), use FormData like regular properties
    if (data.isAsset) {
      const formData = new FormData();
      
      const propertyData = {
        title: data.title,
        description: data.description || '',
        property_type: data.property_type || 'apartment',
        listing_type: 'sell',
        address: data.address,
        city: data.city,
        state: data.state || 'Karnataka',
        pincode: data.pincode || '560001',
        locality: data.locality || '',
        built_up_area: data.built_up_area || null,
        bedrooms: data.bedrooms || null,
        bathrooms: data.bathrooms || null,
        status: 'pending',
        email: data.email || localStorage.getItem('sellerEmail') ,
        fullname: data.fullname || '',
        property_manager_contact: data.property_manager_contact || '',
        sale_price: data.sale_price || 0,
        listed_by_id: sellerToken ? localStorage.getItem('sellerId') : null,
        listed_by_type: data.listed_by_type || 'seller',
        isAsset: true,
      };
      
      // Add property data as JSON string to FormData
      formData.append('property_data', JSON.stringify(propertyData));
      
      // NO AUTHENTICATION HEADERS FOR ASSET CREATION
      const res = await fetch(`${API_BASE}/properties/`, {
        method: 'POST',
        body: formData
      });
      
      if (!res.ok) {
        const errorText = await res.text();
        console.log('API Error Response:', errorText);
        let error;
        try {
          error = JSON.parse(errorText);
        } catch {
          error = { detail: errorText || 'Failed to create property' };
        }
        const errorMessage = error.detail || error.message || JSON.stringify(error) || `HTTP ${res.status}: ${res.statusText}`;
        throw new Error(errorMessage);
      }
      
      return await res.json();
    }
    
    // Create FormData for multipart upload
    const formData = new FormData();
    
    // Prepare property data
    const propertyData = {
      title: data.title,
      description: data.description || '',
      property_type: data.propertyType || data.type || 'apartment',
      listing_type: data.listingType || data.listingIntent || 'sell',
      address: data.address,
      city: data.city,
      state: data.state || 'Karnataka',
      pincode: String(data.pincode),
      locality: data.locality || data.landmark || '',
      landmark: data.landmark || '',
      built_up_area: data.builtUpArea || data.area || null,
      carpet_area: data.carpetArea || null,
      bedrooms: data.bedrooms || (data.bhk ? parseInt(data.bhk.replace('BHK', '')) : null),
      bathrooms: data.bathrooms || null,
      facing: data.facing || '',
      furnishing: data.furnishing || '',
      property_age: data.propertyAge || '',
      possession_status: data.possessionStatus || '',
      sale_price: (data.listingIntent === 'rent' || data.listingType === 'rent') ? null : (data.salePrice || data.price || null),
      monthly_rent: (data.listingIntent === 'rent' || data.listingType === 'rent') ? (data.monthlyRent || data.price || null) : null,
      security_deposit: data.securityDeposit || null,
      amenities: Array.isArray(data.amenities) ? data.amenities : [],
      nearby_places: Array.isArray(data.nearbyPlaces) ? data.nearbyPlaces : [],
      latitude: data.latitude ? parseFloat(data.latitude) : null,
      longitude: data.longitude ? parseFloat(data.longitude) : null,
      virtual_tour_url: data.virtualTourUrl || data.virtualTour || '',
      status: 'pending',
      email: data.email || '',
      fullname: data.fullname || ''
    };
    
    // Add property data as JSON string
    formData.append('property_data', JSON.stringify(propertyData));
    
    // Add image files (convert blob URLs to actual files if needed)
    if (data.images && Array.isArray(data.images)) {
      for (const image of data.images) {
        if (image instanceof File) {
          // Direct File object
          formData.append('images', image);
        } else if (image.file && image.file instanceof File) {
          // File object in wrapper
          formData.append('images', image.file);
        } else if (image.url && image.url.startsWith('blob:')) {
          // Convert blob URL to file
          try {
            const response = await fetch(image.url);
            const blob = await response.blob();
            const file = new File([blob], image.alt_text || 'image.jpg', { type: blob.type });
            formData.append('images', file);
          } catch (e) {
            console.warn('Failed to convert blob URL to file:', e);
          }
        }
      }
    }
    
    // Add video file
    if (data.videos && Array.isArray(data.videos) && data.videos[0]) {
      const video = data.videos[0];
      if (video instanceof File) {
        // Direct File object
        formData.append('video', video);
      } else if (video.file && video.file instanceof File) {
        // File object in wrapper
        formData.append('video', video.file);
      } else if (video.url && video.url.startsWith('blob:')) {
        try {
          const response = await fetch(video.url);
          const blob = await response.blob();
          const file = new File([blob], 'video.mp4', { type: blob.type });
          formData.append('video', file);
        } catch (e) {
          console.warn('Failed to convert video blob URL to file:', e);
        }
      }
    }
    
    // NO AUTHENTICATION HEADERS FOR PROPERTY CREATION
    const res = await fetch(`${API_BASE}/properties/`, {
      method: 'POST',
      body: formData
    });
    
    if (!res.ok) {
      const errorText = await res.text();
      let error;
      try {
        error = JSON.parse(errorText);
      } catch {
        error = { detail: errorText || 'Failed to create property' };
      }
      throw new Error(error.detail || error.message || `HTTP ${res.status}: ${res.statusText}`);
    }
    
    const result = await res.json();
    return result;
  },

  // Fetch properties from FastAPI backend (filtered by logged-in seller)
  fetchProperties: async (userIdOverride?: string): Promise<Property[]> => {
    // Get seller ID from localStorage or use override
    const sellerId = userIdOverride || localStorage.getItem('sellerId');
    
    // Build URL with listed_by_id filter to get only this seller's properties
    const url = sellerId 
      ? `${API_BASE}/properties/?listed_by_id=${sellerId}`
      : `${API_BASE}/properties/`;
    
    console.log("🔍 fetchProperties using sellerId/userId:", sellerId);
    console.log("🔍 fetchProperties URL:", url);
    
    const res = await fetch(url, { 
      method: 'GET'
    });
    if (!res.ok) throw new Error(`Failed to load properties: ${res.status}`);
    const payload = await res.json();
    const rows = Array.isArray(payload) ? payload : (payload.properties ?? []);
    return rows.map(mapRowToProperty);
  },

  // Get single property
  fetchPropertyById: async (id: string | number): Promise<Property | null> => {
    const res = await fetch(`${API_BASE}/properties/${id}/`);
    if (res.status === 404) return null;
    if (!res.ok) throw new Error('Failed to load property');

    const payload = await res.json();
    return mapRowToProperty(payload);
  },

  // Update property
  updateProperty: async (id: string, data: Partial<PropertyFormData>): Promise<void> => {
    const payload = {
      title: data.title,
      description: data.description || '',
      property_type: data.type,
      listing_type: data.listingIntent,
      address: data.address,
      city: data.city,
      state: data.state || 'Karnataka',
      pincode: data.pincode,
      locality: data.landmark || '',
      landmark: data.landmark || '',
      built_up_area: data.area,
      bedrooms: data.bhk ? parseInt(data.bhk.replace('BHK', '')) : null,
      sale_price: data.listingIntent === 'rent' ? null : data.price,
      monthly_rent: data.listingIntent === 'rent' ? data.price : null,
      amenities: data.amenities,
      status: data.status || 'pending',
      fullname: data.fullname,
      email: data.email,
      property_manager_contact: data.property_manager_contact
    };

    const res = await fetch(`${API_BASE}/properties/${id}/`, {
      method: 'PATCH',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(payload)
    });

    if (!res.ok) {
      throw new Error('Failed to update property');
    }
  },

  // Delete property
  deleteProperty: async (id: string): Promise<void> => {
    const res = await fetch(`${API_BASE}/properties/${id}/`, {
      method: 'DELETE'
    });

    if (!res.ok) {
      throw new Error('Failed to delete property');
    }
  },

  // List asset for sale
  listAssetForSale: async (id: string): Promise<{property: Property, seller_details: any}> => {
    console.log('🚀 Starting listAssetForSale for property ID:', id);
    
    const sellerToken = localStorage.getItem('sellerToken');
    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
    };
    
    if (sellerToken) {
      headers['Authorization'] = `Bearer ${sellerToken}`;
    }

    console.log('📤 Calling list-for-sale endpoint with empty body');

    const res = await fetch(`${API_BASE}/properties/${id}/list-for-sale`, {
      method: 'POST',
      headers,
      body: JSON.stringify({}) // Send empty body - backend will fetch seller details
    });

    if (!res.ok) {
      const errorText = await res.text();
      console.error('❌ API Error:', errorText);
      let error;
      try {
        error = JSON.parse(errorText);
      } catch {
        error = { detail: errorText || 'Failed to list asset for sale' };
      }
      throw new Error(error.detail || error.message || 'Failed to list asset for sale');
    }

    const result = await res.json();
    console.log('✅ API Response - Property updated with seller details:', result);
    
    return {
      property: mapRowToProperty(result.property),
      seller_details: result.seller_details || {}
    };
  },

  // Fetch user details
  fetchUserDetails: async (): Promise<{ firstname: string; lastname: string; mobile: string; email: string } | null> => {
    const sellerId = localStorage.getItem('sellerId');
    const sellerToken = localStorage.getItem('sellerToken');
    if (!sellerId || !sellerToken) {
      // Fallback to localStorage values if available
      const storedEmail = localStorage.getItem('sellerEmail');
      const storedName = localStorage.getItem('sellerName');
      const storedMobile = localStorage.getItem('sellerMobile');
      
      if (storedEmail || storedName) {
        const nameParts = (storedName || '').split(' ');
        return {
          firstname: nameParts[0] || '',
          lastname: nameParts.slice(1).join(' ') || '',
          mobile: storedMobile || '',
          email: storedEmail || ''
        };
      }
      return null;
    }

    try {
      const headers: Record<string, string> = {
        'Authorization': `Bearer ${sellerToken}`
      };

      // Try to fetch from seller profile first
      const sellerProfileRes = await fetch(`${API_BASE}/seller-profile/me`, {
        headers
      });
      
      if (sellerProfileRes.ok) {
        const sellerProfile = await sellerProfileRes.json();
        // Parse full_name into firstname and lastname
        const nameParts = (sellerProfile.full_name || '').split(' ');
        return {
          firstname: nameParts[0] || '',
          lastname: nameParts.slice(1).join(' ') || '',
          mobile: sellerProfile.phone_number || '',
          email: sellerProfile.email || ''
        };
      }

      // Fallback to old seller profile endpoint
      const res = await fetch(`${API_BASE}/seller/profile`, {
        headers
      });
      
      if (!res.ok) {
        // If API fails, try localStorage fallback
        const storedEmail = localStorage.getItem('sellerEmail');
        const storedName = localStorage.getItem('sellerName');
        const storedMobile = localStorage.getItem('sellerMobile');
        
        if (storedEmail || storedName) {
          const nameParts = (storedName || '').split(' ');
          return {
            firstname: nameParts[0] || '',
            lastname: nameParts.slice(1).join(' ') || '',
            mobile: storedMobile || '',
            email: storedEmail || ''
          };
        }
        return null;
      }

      const data = await res.json();
      return {
        firstname: data.firstname || data.first_name || '',
        lastname: data.lastname || data.last_name || '',
        mobile: data.mobile || data.phone || data.phone_number || '',
        email: data.email || ''
      };
    } catch (error) {
      console.error('Error fetching user details:', error);
      // Try localStorage fallback on error
      const storedEmail = localStorage.getItem('sellerEmail');
      const storedName = localStorage.getItem('sellerName');
      const storedMobile = localStorage.getItem('sellerMobile');
      
      if (storedEmail || storedName) {
        const nameParts = (storedName || '').split(' ');
        return {
          firstname: nameParts[0] || '',
          lastname: nameParts.slice(1).join(' ') || '',
          mobile: storedMobile || '',
          email: storedEmail || ''
        };
      }
      return null;
    }
  },

  // Get property legal health (no authentication required)
  getPropertyLegalHealth: async (propertyId: string | number): Promise<any> => {
    const res = await fetch(`${API_BASE}/legal-health/property/${propertyId}/legal-health`, {
      method: 'GET',
      headers: {
        'accept': 'application/json'
      }
    });

    if (!res.ok) {
      const errorText = await res.text();
      let error;
      try {
        error = JSON.parse(errorText);
      } catch {
        error = { detail: errorText || 'Failed to get legal health' };
      }
      throw new Error(error.message || error.detail || `HTTP ${res.status}: ${res.statusText}`);
    }

    return await res.json();
  },

  // Cleanup duplicate properties
  cleanupDuplicates: async (): Promise<any> => {
    const sellerId = localStorage.getItem('sellerId');
    if (!sellerId) throw new Error('Seller ID not found');

    const res = await fetch(`${API_BASE}/properties/cleanup-duplicates/${sellerId}`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' }
    });

    if (!res.ok) throw new Error('Failed to cleanup duplicates');
    return await res.json();
  }
};