# System Health Monitoring - Frontend Integration Guide

## Overview
This folder contains a complete React-based system health monitoring dashboard that can be integrated into any React project.

## Features
- Real-time system health monitoring
- Interactive metrics charts
- System alerts management
- System logs viewer
- Service health checks
- Responsive design
- TypeScript support

## Folder Structure
```
frontend-system-health-monitoring/
├── components/
│   ├── SystemHealthDashboard.tsx    # Main dashboard component
│   ├── MetricsChart.tsx            # Metrics visualization
│   ├── AlertsPanel.tsx             # Alerts management
│   ├── LogsPanel.tsx               # System logs viewer
│   └── HealthChecks.tsx            # Service health status
├── hooks/
│   └── useSystemMonitoring.ts      # Custom React hooks
├── types/
│   └── index.ts                    # TypeScript interfaces
├── utils/
│   └── api.ts                      # API service layer
├── styles/
│   └── SystemHealthDashboard.css   # Complete styling
└── README.md                       # This file
```

## Dependencies Required

Add these to your project's package.json:

```json
{
  "dependencies": {
    "react": "^18.0.0",
    "react-dom": "^18.0.0",
    "react-icons": "^4.8.0"
  },
  "devDependencies": {
    "@types/react": "^18.0.0",
    "@types/react-dom": "^18.0.0",
    "typescript": "^4.9.0"
  }
}
```

## Installation Steps

### 1. Copy Files
Copy the entire `frontend-system-health-monitoring` folder to your React project's `src` directory.

### 2. Install Dependencies
```bash
npm install react-icons
# or
yarn add react-icons
```

### 3. Configure API URL
Edit `utils/api.ts` and update the API_BASE_URL:

```typescript
const API_BASE_URL = process.env.REACT_APP_API_URL || 'http://your-backend-url/api/system_monitoring';
```

### 4. Add Environment Variable
Create/update your `.env` file:
```
REACT_APP_API_URL=http://localhost:8000/api/system_monitoring
```

### 5. Import and Use Components

#### Option A: Use Complete Dashboard
```tsx
import React from 'react';
import SystemHealthDashboard from './frontend-system-health-monitoring/components/SystemHealthDashboard';

function App() {
  return (
    <div className="App">
      <SystemHealthDashboard />
    </div>
  );
}

export default App;
```

#### Option B: Use Individual Components
```tsx
import React from 'react';
import AlertsPanel from './frontend-system-health-monitoring/components/AlertsPanel';
import MetricsChart from './frontend-system-health-monitoring/components/MetricsChart';

function MyDashboard() {
  return (
    <div>
      <MetricsChart />
      <AlertsPanel />
    </div>
  );
}
```

### 6. Add to Router (if using React Router)
```tsx
import { BrowserRouter as Router, Routes, Route } from 'react-router-dom';
import SystemHealthDashboard from './frontend-system-health-monitoring/components/SystemHealthDashboard';

function App() {
  return (
    <Router>
      <Routes>
        <Route path="/system-health" element={<SystemHealthDashboard />} />
      </Routes>
    </Router>
  );
}
```

## Customization

### 1. Styling
- Modify `styles/SystemHealthDashboard.css` to match your theme
- Override CSS variables or classes as needed
- The design is fully responsive

### 2. API Configuration
- Update `utils/api.ts` to match your backend endpoints
- Modify authentication headers if needed
- Add error handling as required

### 3. Components
- Each component is modular and can be used independently
- Customize refresh intervals in hooks
- Add additional metrics or filters as needed

## Usage Examples

### Basic Usage
```tsx
import SystemHealthDashboard from './frontend-system-health-monitoring/components/SystemHealthDashboard';

// Use the complete dashboard
<SystemHealthDashboard />
```

### Custom Refresh Intervals
```tsx
import { useSystemHealth } from './frontend-system-health-monitoring/hooks/useSystemMonitoring';

function CustomHealthWidget() {
  const { health, loading, error } = useSystemHealth(10000); // 10 second refresh
  
  if (loading) return <div>Loading...</div>;
  if (error) return <div>Error: {error}</div>;
  
  return (
    <div>
      <h3>System Health: {health?.overall_health}</h3>
      <p>Score: {health?.health_score}%</p>
    </div>
  );
}
```

### Filtered Alerts
```tsx
import { useSystemAlerts } from './frontend-system-health-monitoring/hooks/useSystemMonitoring';

function CriticalAlerts() {
  const { alerts, loading } = useSystemAlerts({ severity: 'critical', status: 'active' });
  
  return (
    <div>
      <h3>Critical Alerts ({alerts.length})</h3>
      {alerts.map(alert => (
        <div key={alert.id}>{alert.title}</div>
      ))}
    </div>
  );
}
```

## Integration with Different Frameworks

### Next.js
```tsx
// pages/system-health.tsx
import dynamic from 'next/dynamic';

const SystemHealthDashboard = dynamic(
  () => import('../components/frontend-system-health-monitoring/components/SystemHealthDashboard'),
  { ssr: false }
);

export default function SystemHealthPage() {
  return <SystemHealthDashboard />;
}
```

### Vite
```tsx
// No special configuration needed, works out of the box
import SystemHealthDashboard from './frontend-system-health-monitoring/components/SystemHealthDashboard';
```

## Troubleshooting

### Common Issues

1. **CORS Errors**
   - Ensure your backend allows requests from your frontend domain
   - Check CORS configuration in Django settings

2. **API Connection Issues**
   - Verify the API_BASE_URL is correct
   - Check if backend is running and accessible
   - Verify network connectivity

3. **Styling Issues**
   - Ensure CSS file is imported
   - Check for CSS conflicts with existing styles
   - Verify react-icons is installed

4. **TypeScript Errors**
   - Ensure all type definitions are imported correctly
   - Check TypeScript configuration
   - Verify React types are installed

### Performance Optimization

1. **Reduce API Calls**
   - Increase refresh intervals for less critical data
   - Implement caching if needed

2. **Component Optimization**
   - Use React.memo for expensive components
   - Implement virtual scrolling for large log lists

3. **Bundle Size**
   - Import only needed icons from react-icons
   - Consider code splitting for large dashboards

## Support

For issues or questions:
1. Check the backend integration guide
2. Verify API endpoints are working
3. Check browser console for errors
4. Ensure all dependencies are installed

## License
This component library is provided as-is for integration into your projects.