import React, { useState } from 'react';
import { FaExclamationTriangle, FaCheck, FaEye, FaSync } from 'react-icons/fa';
import { useSystemAlerts } from '../hooks/useSystemMonitoring';

const AlertsPanel: React.FC = () => {
  const [filters, setFilters] = useState({
    severity: '',
    status: '',
    page: 1
  });

  const { alerts, pagination, loading, acknowledgeAlert, resolveAlert, refresh } = useSystemAlerts(filters);

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case 'critical': return '#e74c3c';
      case 'error': return '#e67e22';
      case 'warning': return '#f1c40f';
      case 'info': return '#3498db';
      default: return '#95a5a6';
    }
  };

  const formatTimestamp = (timestamp: string) => {
    return new Date(timestamp).toLocaleString();
  };

  return (
    <div className="alerts-panel">
      <div className="panel-header">
        <h2><FaExclamationTriangle /> System Alerts</h2>
        <button onClick={refresh} className="refresh-btn">
          <FaSync className={loading ? 'spinning' : ''} />
          Refresh
        </button>
      </div>

      <div className="filters-bar">
        <select 
          value={filters.severity} 
          onChange={(e) => setFilters({...filters, severity: e.target.value, page: 1})}
        >
          <option value="">All Severities</option>
          <option value="critical">Critical</option>
          <option value="error">Error</option>
          <option value="warning">Warning</option>
          <option value="info">Info</option>
        </select>
        
        <select 
          value={filters.status} 
          onChange={(e) => setFilters({...filters, status: e.target.value, page: 1})}
        >
          <option value="">All Statuses</option>
          <option value="active">Active</option>
          <option value="acknowledged">Acknowledged</option>
          <option value="resolved">Resolved</option>
        </select>
      </div>

      <div className="alerts-list">
        {loading ? (
          <div className="loading">Loading alerts...</div>
        ) : !alerts || alerts.length === 0 ? (
          <div className="empty-state">
            <FaCheck size={48} />
            <h3>No Alerts</h3>
            <p>All systems are running smoothly</p>
          </div>
        ) : (
          (alerts || []).map((alert) => (
            <div key={alert.id} className={`alert-card ${alert.severity} ${alert.status}`}>
              <div className="alert-header">
                <div className="alert-info">
                  <span 
                    className="severity-badge"
                    style={{ backgroundColor: getSeverityColor(alert.severity) }}
                  >
                    {alert.severity.toUpperCase()}
                  </span>
                  <h4 className="alert-title">{alert.title}</h4>
                  <span className="alert-time">{formatTimestamp(alert.created_at)}</span>
                </div>
                
                <div className="alert-actions">
                  {alert.status === 'active' && (
                    <>
                      <button 
                        onClick={() => acknowledgeAlert(alert.id)}
                        className="action-btn acknowledge"
                      >
                        <FaEye />
                      </button>
                      <button 
                        onClick={() => resolveAlert(alert.id)}
                        className="action-btn resolve"
                      >
                        <FaCheck />
                      </button>
                    </>
                  )}
                  <span className={`status-badge ${alert.status}`}>
                    {alert.status.toUpperCase()}
                  </span>
                </div>
              </div>
              
              <div className="alert-message">
                {alert.message}
              </div>
              
              <div className="alert-details">
                {alert.metric_type && <span>Metric: {alert.metric_type}</span>}
                {alert.threshold_value && <span>Threshold: {alert.threshold_value}</span>}
                {alert.current_value && <span>Current: {alert.current_value}</span>}
              </div>
            </div>
          ))
        )}
      </div>

      {pagination && pagination.total_pages > 1 && (
        <div className="pagination">
          <button 
            disabled={!pagination.has_previous}
            onClick={() => setFilters({...filters, page: filters.page - 1})}
          >
            Previous
          </button>
          <span>Page {pagination.current_page} of {pagination.total_pages}</span>
          <button 
            disabled={!pagination.has_next}
            onClick={() => setFilters({...filters, page: filters.page + 1})}
          >
            Next
          </button>
        </div>
      )}
    </div>
  );
};

export default AlertsPanel;