import React, { useState, useEffect } from 'react';
import { FaShieldAlt, FaCheckCircle, FaExclamationCircle, FaTimesCircle, FaSync, FaClock } from 'react-icons/fa';
import { SystemMonitoringAPI } from '../utils/api';
import { HealthCheck } from '../types';

const HealthChecks: React.FC = () => {
  const [healthChecks, setHealthChecks] = useState<HealthCheck[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchHealthChecks = async () => {
    try {
      setLoading(true);
      const data = await SystemMonitoringAPI.getHealthChecks();
      setHealthChecks(data);
      setError(null);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch health checks');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchHealthChecks();
    const interval = setInterval(fetchHealthChecks, 30000);
    return () => clearInterval(interval);
  }, []);

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'healthy': return <FaCheckCircle />;
      case 'degraded': return <FaExclamationCircle />;
      case 'unhealthy': return <FaTimesCircle />;
      default: return <FaExclamationCircle />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'healthy': return '#27ae60';
      case 'degraded': return '#f39c12';
      case 'unhealthy': return '#e74c3c';
      default: return '#95a5a6';
    }
  };

  const formatTimestamp = (timestamp: string) => {
    return new Date(timestamp).toLocaleString();
  };

  const healthyCount = (healthChecks || []).filter(check => check.status === 'healthy').length;
  const totalCount = (healthChecks || []).length;
  const healthPercentage = totalCount > 0 ? (healthyCount / totalCount) * 100 : 0;

  return (
    <div className="health-checks">
      <div className="panel-header">
        <h2><FaShieldAlt /> Service Health Checks</h2>
        <button onClick={fetchHealthChecks} className="refresh-btn">
          <FaSync className={loading ? 'spinning' : ''} />
          Refresh
        </button>
      </div>

      <div className="health-summary">
        <div className="summary-card">
          <div className="summary-icon">
            <FaShieldAlt />
          </div>
          <div className="summary-content">
            <h3>{healthyCount} / {totalCount}</h3>
            <p>Services Healthy</p>
            <div className="health-bar">
              <div 
                className="health-fill"
                style={{ 
                  width: `${healthPercentage}%`,
                  backgroundColor: healthPercentage > 80 ? '#27ae60' : healthPercentage > 60 ? '#f39c12' : '#e74c3c'
                }}
              ></div>
            </div>
          </div>
        </div>
      </div>

      {error && (
        <div className="error-message">
          <FaExclamationCircle />
          <span>{error}</span>
        </div>
      )}

      <div className="health-checks-grid">
        {loading ? (
          <div className="loading">Loading health checks...</div>
        ) : !healthChecks || healthChecks.length === 0 ? (
          <div className="empty-state">
            <FaShieldAlt size={48} />
            <h3>No Health Checks</h3>
            <p>No service health checks configured</p>
          </div>
        ) : (
          (healthChecks || []).map((check, index) => (
            <div key={index} className={`health-check-card ${check.status}`}>
              <div className="check-header">
                <div className="service-info">
                  <div 
                    className="status-icon"
                    style={{ color: getStatusColor(check.status) }}
                  >
                    {getStatusIcon(check.status)}
                  </div>
                  <h4 className="service-name">{check.service_name}</h4>
                </div>
                
                <span 
                  className={`status-badge ${check.status}`}
                  style={{ backgroundColor: getStatusColor(check.status) }}
                >
                  {check.status.toUpperCase()}
                </span>
              </div>

              <div className="check-details">
                <div className="detail-row">
                  <span className="detail-label">
                    <FaClock /> Last Check:
                  </span>
                  <span className="detail-value">
                    {formatTimestamp(check.last_check)}
                  </span>
                </div>

                {check.response_time_ms && (
                  <div className="detail-row">
                    <span className="detail-label">Response Time:</span>
                    <span className="detail-value response-time">
                      {check.response_time_ms.toFixed(0)}ms
                    </span>
                  </div>
                )}

                {check.error_message && (
                  <div className="error-details">
                    <span className="error-label">Error:</span>
                    <span className="error-text">{check.error_message}</span>
                  </div>
                )}
              </div>
            </div>
          ))
        )}
      </div>
    </div>
  );
};

export default HealthChecks;