import React, { useState } from 'react';
import { FaList, FaSearch, FaSync, FaExclamationCircle, FaInfoCircle, FaExclamationTriangle } from 'react-icons/fa';
import { useSystemLogs } from '../hooks/useSystemMonitoring';

const LogsPanel: React.FC = () => {
  const [filters, setFilters] = useState({
    level: '',
    source: '',
    search: '',
    page: 1
  });

  const { logs, pagination, loading, refresh } = useSystemLogs(filters);

  const getLevelIcon = (level: string) => {
    switch (level) {
      case 'error':
      case 'critical':
        return <FaExclamationCircle />;
      case 'warning':
        return <FaExclamationTriangle />;
      default:
        return <FaInfoCircle />;
    }
  };

  const getLevelColor = (level: string) => {
    switch (level) {
      case 'critical': return '#e74c3c';
      case 'error': return '#e67e22';
      case 'warning': return '#f1c40f';
      case 'info': return '#3498db';
      case 'debug': return '#95a5a6';
      default: return '#95a5a6';
    }
  };

  const formatTimestamp = (timestamp: string) => {
    return new Date(timestamp).toLocaleString();
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setFilters({...filters, page: 1});
  };

  return (
    <div className="logs-panel">
      <div className="panel-header">
        <h2><FaList /> System Logs</h2>
        <button onClick={refresh} className="refresh-btn">
          <FaSync className={loading ? 'spinning' : ''} />
          Refresh
        </button>
      </div>

      <div className="filters-bar">
        <form onSubmit={handleSearch} className="search-form">
          <div className="search-input">
            <FaSearch />
            <input 
              type="text" 
              placeholder="Search logs..." 
              value={filters.search}
              onChange={(e) => setFilters({...filters, search: e.target.value})}
            />
          </div>
          <button type="submit" className="search-btn">Search</button>
        </form>

        <div className="filter-group">
          <select 
            value={filters.level} 
            onChange={(e) => setFilters({...filters, level: e.target.value, page: 1})}
          >
            <option value="">All Levels</option>
            <option value="critical">Critical</option>
            <option value="error">Error</option>
            <option value="warning">Warning</option>
            <option value="info">Info</option>
            <option value="debug">Debug</option>
          </select>
          
          <input 
            type="text" 
            placeholder="Source..." 
            value={filters.source}
            onChange={(e) => setFilters({...filters, source: e.target.value, page: 1})}
          />
        </div>
      </div>

      <div className="logs-list">
        {loading ? (
          <div className="loading">Loading logs...</div>
        ) : !logs || logs.length === 0 ? (
          <div className="empty-state">
            <FaList size={48} />
            <h3>No Logs Found</h3>
            <p>No logs match your current filters</p>
          </div>
        ) : (
          <div className="logs-table">
            {(logs || []).map((log) => (
              <div key={log.id} className={`log-row ${log.level}`}>
                <div className="log-time">
                  {formatTimestamp(log.timestamp)}
                </div>
                
                <div className="log-level">
                  <span 
                    className="level-badge"
                    style={{ backgroundColor: getLevelColor(log.level) }}
                  >
                    {getLevelIcon(log.level)}
                    {log.level.toUpperCase()}
                  </span>
                </div>
                
                <div className="log-source">
                  <span className="source-tag">{log.source}</span>
                </div>
                
                <div className="log-message">
                  <span className="message-text">{log.message}</span>
                </div>
                
                <div className="log-details">
                  {log.user_id && <span className="detail-item">User: {log.user_id}</span>}
                  {log.ip_address && <span className="detail-item">IP: {log.ip_address}</span>}
                  {log.request_path && <span className="detail-item">Path: {log.request_path}</span>}
                </div>
              </div>
            ))}
          </div>
        )}
      </div>

      {pagination && pagination.total_pages > 1 && (
        <div className="pagination">
          <button 
            disabled={!pagination.has_previous}
            onClick={() => setFilters({...filters, page: filters.page - 1})}
          >
            Previous
          </button>
          <span>Page {pagination.current_page} of {pagination.total_pages}</span>
          <span className="total-count">({pagination.total_count} total logs)</span>
          <button 
            disabled={!pagination.has_next}
            onClick={() => setFilters({...filters, page: filters.page + 1})}
          >
            Next
          </button>
        </div>
      )}
    </div>
  );
};

export default LogsPanel;