import React, { useState } from 'react';
import { 
  FaHeartbeat, FaExclamationTriangle, FaServer, FaChartLine,
  FaMicrochip, FaMemory, FaHdd, FaDatabase, FaShieldAlt, FaList
} from 'react-icons/fa';
import '../styles/SystemHealthDashboard.css';

const SystemHealthDashboard: React.FC = () => {
  const [activeTab, setActiveTab] = useState('overview');

  const health = {
    overall_health: 'healthy',
    health_score: 95
  };

  const stats = {
    system_metrics: {
      cpu_usage: 25.5,
      memory_usage: 45.2,
      disk_usage: 60.1
    },
    alert_counts: { critical: 0, error: 0, warning: 0, info: 0 },
    service_health: { health_percentage: 100 }
  };

  const getHealthColor = (status: string) => {
    return status === 'healthy' ? '#27ae60' : '#95a5a6';
  };

  const getSeverityColor = (severity: string) => {
    const colors: any = {
      critical: '#e74c3c',
      error: '#e67e22',
      warning: '#f1c40f',
      info: '#3498db'
    };
    return colors[severity] || '#95a5a6';
  };

  return (
    <div className="system-health-dashboard">
      <div className="dashboard-header">
        <div className="header-content">
          <h1><FaHeartbeat /> System Health & Monitoring</h1>
          <p>Real-time system performance and health monitoring</p>
        </div>
        <div className="health-indicator">
          <div 
            className={`health-status ${health.overall_health}`}
            style={{ backgroundColor: getHealthColor(health.overall_health) }}
          >
            <span className="health-text">{health.overall_health.toUpperCase()}</span>
            <span className="health-score">{health.health_score}%</span>
          </div>
        </div>
      </div>

      <div className="dashboard-tabs">
        <button 
          className={`tab ${activeTab === 'overview' ? 'active' : ''}`}
          onClick={() => setActiveTab('overview')}
        >
          <FaChartLine /> Overview
        </button>
        <button 
          className={`tab ${activeTab === 'metrics' ? 'active' : ''}`}
          onClick={() => setActiveTab('metrics')}
        >
          <FaServer /> Metrics
        </button>
        <button 
          className={`tab ${activeTab === 'alerts' ? 'active' : ''}`}
          onClick={() => setActiveTab('alerts')}
        >
          <FaExclamationTriangle /> Alerts
        </button>
        <button 
          className={`tab ${activeTab === 'logs' ? 'active' : ''}`}
          onClick={() => setActiveTab('logs')}
        >
          <FaList /> Logs
        </button>
        <button 
          className={`tab ${activeTab === 'health' ? 'active' : ''}`}
          onClick={() => setActiveTab('health')}
        >
          <FaShieldAlt /> Health Checks
        </button>
      </div>

      {activeTab === 'overview' && (
        <div className="overview-section">
          <div className="stats-grid">
            <div className="stat-card">
              <div className="stat-icon cpu">
                <FaMicrochip />
              </div>
              <div className="stat-content">
                <h3>{stats.system_metrics.cpu_usage.toFixed(1)}%</h3>
                <p>CPU Usage</p>
              </div>
            </div>

            <div className="stat-card">
              <div className="stat-icon memory">
                <FaMemory />
              </div>
              <div className="stat-content">
                <h3>{stats.system_metrics.memory_usage.toFixed(1)}%</h3>
                <p>Memory Usage</p>
              </div>
            </div>

            <div className="stat-card">
              <div className="stat-icon disk">
                <FaHdd />
              </div>
              <div className="stat-content">
                <h3>{stats.system_metrics.disk_usage.toFixed(1)}%</h3>
                <p>Disk Usage</p>
              </div>
            </div>

            <div className="stat-card">
              <div className="stat-icon database">
                <FaDatabase />
              </div>
              <div className="stat-content">
                <h3>{stats.service_health.health_percentage.toFixed(0)}%</h3>
                <p>Services Healthy</p>
              </div>
            </div>
          </div>

          <div className="alerts-summary">
            <h3>Active Alerts Summary</h3>
            <div className="alert-counts">
              {Object.entries(stats.alert_counts).map(([severity, count]) => (
                <div key={severity} className="alert-count-item">
                  <div 
                    className="alert-dot"
                    style={{ backgroundColor: getSeverityColor(severity) }}
                  ></div>
                  <span className="alert-label">{severity}</span>
                  <span className="alert-number">{count}</span>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      {activeTab === 'metrics' && <div style={{padding: '20px'}}>Metrics Chart Coming Soon</div>}
      {activeTab === 'alerts' && <div style={{padding: '20px'}}>No Active Alerts</div>}
      {activeTab === 'logs' && <div style={{padding: '20px'}}>No Recent Logs</div>}
      {activeTab === 'health' && <div style={{padding: '20px'}}>All Services Healthy</div>}
    </div>
  );
};

export default SystemHealthDashboard;
