import React from 'react';
import { TimelineEvent } from '@/types/transaction';
import { CheckCircle, XCircle, Clock, AlertTriangle, Info } from 'lucide-react';

interface TimelineViewProps {
  events: TimelineEvent[];
}

const statusIcons = {
  success: <CheckCircle className="w-5 h-5 text-green-600" />,
  failed: <XCircle className="w-5 h-5 text-red-600" />,
  pending: <Clock className="w-5 h-5 text-yellow-600" />,
  warning: <AlertTriangle className="w-5 h-5 text-orange-600" />,
  info: <Info className="w-5 h-5 text-blue-600" />,
};

export const TimelineView: React.FC<TimelineViewProps> = ({ events }) => {
  if (events.length === 0) {
    return <div className="text-center py-4 text-gray-500">No timeline events</div>;
  }

  return (
    <div className="relative">
      {/* Timeline line */}
      <div className="absolute left-6 top-0 bottom-0 w-0.5 bg-gray-200" />
      
      <div className="space-y-6">
        {events.map((event, index) => (
          <div key={event.id} className="relative flex gap-4">
            {/* Icon */}
            <div className="relative z-10 flex items-center justify-center w-12 h-12 bg-white border-2 border-gray-200 rounded-full">
              {statusIcons[event.event_status]}
            </div>
            
            {/* Content */}
            <div className="flex-1 pb-6">
              <div className="bg-white p-4 rounded-lg border border-gray-200 shadow-sm">
                <div className="flex justify-between items-start mb-2">
                  <h4 className="font-semibold text-gray-900">{event.event_title}</h4>
                  <span className="text-xs text-gray-500">
                    {new Date(event.event_timestamp).toLocaleString()}
                  </span>
                </div>
                
                {event.event_description && (
                  <p className="text-sm text-gray-600 mt-1">{event.event_description}</p>
                )}
                
                {(event.actor_type || event.actor_id) && (
                  <div className="mt-2 flex items-center gap-2">
                    <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-700">
                      {event.actor_type === 'user' && '👤 User'}
                      {event.actor_type === 'admin' && '👨‍💼 Admin'}
                      {event.actor_type === 'system' && '🤖 System'}
                      {!event.actor_type && 'System'}
                    </span>
                    {event.actor_id && (
                      <span className="text-xs text-gray-500">ID: {event.actor_id.substring(0, 8)}...</span>
                    )}
                  </div>
                )}
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
};
