import React, { useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import { transactionApi } from '@/api/transactions';
import { TransactionType, PaymentMethod } from '@/types/transaction';
import { toast } from 'sonner';

interface PaymentModalProps {
  isOpen: boolean;
  onClose: () => void;
  amount: number;
  propertyId?: number;
  transactionType: TransactionType;
  description: string;
  onSuccess?: (transactionId: string) => void;
}

export const PaymentModal: React.FC<PaymentModalProps> = ({
  isOpen,
  onClose,
  amount,
  propertyId,
  transactionType,
  description,
  onSuccess
}) => {
  const [selectedMethod, setSelectedMethod] = useState<PaymentMethod>(PaymentMethod.UPI);
  const [processing, setProcessing] = useState(false);

  const paymentMethods = [
    { value: PaymentMethod.UPI, label: 'UPI', icon: '📱' },
    { value: PaymentMethod.CREDIT_CARD, label: 'Credit Card', icon: '💳' },
    { value: PaymentMethod.DEBIT_CARD, label: 'Debit Card', icon: '💳' },
    { value: PaymentMethod.NET_BANKING, label: 'Net Banking', icon: '🏦' },
    { value: PaymentMethod.WALLET, label: 'Wallet', icon: '👛' },
  ];

  const handlePayment = async () => {
    setProcessing(true);
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      
      const transaction = await transactionApi.create({
        user_id: user.user_id,
        property_id: propertyId,
        transaction_type: transactionType,
        payment_method: selectedMethod,
        amount: amount,
        description: description
      });

      if (transaction.transaction_status === 'completed') {
        toast.success('Payment successful!');
        onSuccess?.(transaction.id);
        onClose();
      } else {
        toast.error('Payment failed. Please try again.');
      }
    } catch (error) {
      console.error('Payment error:', error);
      toast.error('Payment failed. Please try again.');
    } finally {
      setProcessing(false);
    }
  };

  const tax = amount * 0.18;
  const processingFee = amount * 0.02;
  const total = amount + tax + processingFee;

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-sm">
        <DialogHeader>
          <DialogTitle>Complete Payment</DialogTitle>
        </DialogHeader>

        <div className="space-y-4">
          <div className="bg-gray-50 p-4 rounded-lg space-y-2">
            <div className="flex justify-between text-sm">
              <span>Amount</span>
              <span>₹{amount.toLocaleString()}</span>
            </div>
            <div className="flex justify-between text-sm text-gray-600">
              <span>GST (18%)</span>
              <span>₹{tax.toFixed(2)}</span>
            </div>
            <div className="flex justify-between text-sm text-gray-600">
              <span>Processing Fee (2%)</span>
              <span>₹{processingFee.toFixed(2)}</span>
            </div>
            <div className="border-t pt-2 flex justify-between font-semibold">
              <span>Total</span>
              <span>₹{total.toFixed(2)}</span>
            </div>
          </div>

          <div>
            <Label className="mb-2 block">Select Payment Method</Label>
            <div className="grid grid-cols-2 gap-2">
              {paymentMethods.map((method) => (
                <button
                  key={method.value}
                  type="button"
                  onClick={(e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    setSelectedMethod(method.value);
                  }}
                  disabled={processing}
                  className={`p-3 border-2 rounded-lg text-sm flex items-center gap-2 transition-all cursor-pointer ${
                    selectedMethod === method.value
                      ? 'border-blue-600 bg-blue-100 text-blue-900 font-semibold shadow-md'
                      : 'border-gray-300 hover:border-blue-400 hover:bg-gray-50'
                  } ${processing ? 'opacity-50 cursor-not-allowed' : ''}`}
                >
                  <span>{method.icon}</span>
                  <span>{method.label}</span>
                </button>
              ))}
            </div>
          </div>

          <div className="flex gap-2 pt-4">
            <Button
              variant="outline"
              onClick={onClose}
              disabled={processing}
              className="flex-1"
            >
              Cancel
            </Button>
            <Button
              onClick={handlePayment}
              disabled={processing}
              className="flex-1"
            >
              {processing ? 'Processing...' : 'Pay Now'}
            </Button>
          </div>

          <p className="text-xs text-gray-500 text-center">
            This is a mock payment. No actual charges will be made.
          </p>
        </div>
      </DialogContent>
    </Dialog>
  );
};
