import React from 'react';
import { Transaction, TransactionStatus } from '@/types/transaction';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';

interface TransactionCardProps {
  transaction: Transaction;
  onClick?: () => void;
}

const statusColors: Record<TransactionStatus, string> = {
  [TransactionStatus.INITIATED]: 'bg-blue-100 text-blue-800',
  [TransactionStatus.PROCESSING]: 'bg-yellow-100 text-yellow-800',
  [TransactionStatus.ESCROW_FUNDED]: 'bg-orange-100 text-orange-800',
  [TransactionStatus.COMPLETED]: 'bg-green-100 text-green-800',
  [TransactionStatus.FAILED]: 'bg-red-100 text-red-800',
  [TransactionStatus.REFUNDED]: 'bg-purple-100 text-purple-800',
  [TransactionStatus.CANCELLED]: 'bg-gray-100 text-gray-800',
};

export const TransactionCard: React.FC<TransactionCardProps> = ({ transaction, onClick }) => {
  const getRefundBadgeColor = (status: string) => {
    switch(status) {
      case 'pending': return 'bg-orange-100 text-orange-800';
      case 'rejected': return 'bg-red-100 text-red-800';
      case 'approved': return 'bg-blue-100 text-blue-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };
  
  return (
    <Card className="p-4 hover:shadow-md transition-shadow cursor-pointer" onClick={onClick}>
      <div className="flex justify-between items-start mb-2">
        <div>
          <h3 className="font-semibold text-lg">{transaction.transaction_number}</h3>
          <p className="text-sm text-gray-600">{transaction.description || 'No description'}</p>
        </div>
        <div className="flex flex-col gap-1 items-end">
          <Badge className={statusColors[transaction.transaction_status]}>
            {transaction.transaction_status.toUpperCase()}
          </Badge>
          {transaction.refund_status && (
            <Badge className={getRefundBadgeColor(transaction.refund_status)}>
              Refund: {transaction.refund_status.toUpperCase()}
            </Badge>
          )}
        </div>
      </div>
      
      <div className="grid grid-cols-2 gap-2 mt-3 text-sm">
        <div>
          <span className="text-gray-600">Amount:</span>
          <span className="font-semibold ml-2">₹{transaction.amount.toLocaleString()}</span>
        </div>
        <div>
          <span className="text-gray-600">Total:</span>
          <span className="font-semibold ml-2">₹{transaction.total_amount.toLocaleString()}</span>
        </div>
        <div>
          <span className="text-gray-600">Type:</span>
          <span className="ml-2">{transaction.transaction_type.replace('_', ' ')}</span>
        </div>
        <div>
          <span className="text-gray-600">Method:</span>
          <span className="ml-2">{transaction.payment_method.replace('_', ' ')}</span>
        </div>
      </div>
      
      <div className="mt-3 text-xs text-gray-500">
        {new Date(transaction.created_at).toLocaleString()}
      </div>
    </Card>
  );
};
