import React, { useState, useEffect } from 'react';
import { AlertTriangle, Clock } from 'lucide-react';

interface LockoutMessageProps {
  isLocked: boolean;
  remainingMinutes?: number;
  unlockAt?: string;
  onClose?: () => void;
}

export const LockoutMessage: React.FC<LockoutMessageProps> = ({
  isLocked,
  remainingMinutes = 0,
  unlockAt,
  onClose
}) => {
  const [timeLeft, setTimeLeft] = useState(remainingMinutes);

  useEffect(() => {
    if (!isLocked || timeLeft <= 0) return;

    const timer = setInterval(() => {
      setTimeLeft(prev => {
        if (prev <= 1) {
          clearInterval(timer);
          onClose?.();
          return 0;
        }
        return prev - 1;
      });
    }, 60000); // Update every minute

    return () => clearInterval(timer);
  }, [isLocked, timeLeft, onClose]);

  if (!isLocked) return null;

  const formatTime = (minutes: number) => {
    if (minutes < 60) return `${minutes} minute${minutes !== 1 ? 's' : ''}`;
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    return `${hours} hour${hours !== 1 ? 's' : ''} ${mins > 0 ? `${mins} minute${mins !== 1 ? 's' : ''}` : ''}`;
  };

  return (
    <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-4">
      <div className="flex items-start">
        <AlertTriangle className="h-5 w-5 text-red-500 mt-0.5 mr-3 flex-shrink-0" />
        <div className="flex-1">
          <h3 className="text-sm font-medium text-red-800 mb-1">
            Account Temporarily Locked
          </h3>
          <p className="text-sm text-red-700 mb-2">
            Your account has been locked due to multiple failed login attempts.
          </p>
          <div className="flex items-center text-sm text-red-600">
            <Clock className="h-4 w-4 mr-1" />
            <span>
              Unlocks in: <strong>{formatTime(timeLeft)}</strong>
            </span>
          </div>
          {unlockAt && (
            <p className="text-xs text-red-500 mt-1">
              Unlock time: {new Date(unlockAt).toLocaleString()}
            </p>
          )}
        </div>
      </div>
    </div>
  );
};

interface LoginAttemptsWarningProps {
  remainingAttempts: number;
  maxAttempts?: number;
}

export const LoginAttemptsWarning: React.FC<LoginAttemptsWarningProps> = ({
  remainingAttempts,
  maxAttempts = 5
}) => {
  if (remainingAttempts >= maxAttempts) return null;

  const isUrgent = remainingAttempts <= 2;

  return (
    <div className={`border rounded-lg p-3 mb-4 ${
      isUrgent 
        ? 'bg-red-50 border-red-200' 
        : 'bg-yellow-50 border-yellow-200'
    }`}>
      <div className="flex items-center">
        <AlertTriangle className={`h-4 w-4 mr-2 ${
          isUrgent ? 'text-red-500' : 'text-yellow-500'
        }`} />
        <p className={`text-sm font-medium ${
          isUrgent ? 'text-red-800' : 'text-yellow-800'
        }`}>
          {remainingAttempts} login attempt{remainingAttempts !== 1 ? 's' : ''} remaining
        </p>
      </div>
      <p className={`text-xs mt-1 ${
        isUrgent ? 'text-red-600' : 'text-yellow-600'
      }`}>
        Account will be locked for 30 minutes after {maxAttempts} failed attempts
      </p>
    </div>
  );
};