import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';
import { Progress } from '../ui/progress';
import { ProjectInventory, UNIT_STATUS_COLORS } from '../../types/unitInventory';

interface AvailabilityStatsProps {
  inventory: ProjectInventory;
}

const AvailabilityStats: React.FC<AvailabilityStatsProps> = ({ inventory }) => {
  const stats = [
    {
      label: 'Available',
      count: inventory.available_units,
      percentage: (inventory.available_units / inventory.total_units) * 100,
      color: UNIT_STATUS_COLORS.Available,
      bgColor: 'bg-green-50',
      textColor: 'text-green-700'
    },
    {
      label: 'Booked',
      count: inventory.booked_units,
      percentage: (inventory.booked_units / inventory.total_units) * 100,
      color: UNIT_STATUS_COLORS.Booked,
      bgColor: 'bg-blue-50',
      textColor: 'text-blue-700'
    },
    {
      label: 'Sold',
      count: inventory.sold_units,
      percentage: (inventory.sold_units / inventory.total_units) * 100,
      color: UNIT_STATUS_COLORS.Sold,
      bgColor: 'bg-red-50',
      textColor: 'text-red-700'
    },
    {
      label: 'On Hold',
      count: inventory.on_hold_units,
      percentage: (inventory.on_hold_units / inventory.total_units) * 100,
      color: UNIT_STATUS_COLORS['On Hold'],
      bgColor: 'bg-yellow-50',
      textColor: 'text-yellow-700'
    },
    {
      label: 'Blocked',
      count: inventory.blocked_units,
      percentage: (inventory.blocked_units / inventory.total_units) * 100,
      color: UNIT_STATUS_COLORS.Blocked,
      bgColor: 'bg-gray-50',
      textColor: 'text-gray-700'
    }
  ];

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
      {stats.map((stat) => (
        <Card key={stat.label} className={`${stat.bgColor} border-0`}>
          <CardContent className="p-4">
            <div className="flex items-center justify-between mb-2">
              <div className="flex items-center gap-2">
                <div 
                  className="w-3 h-3 rounded-full"
                  style={{ backgroundColor: stat.color }}
                />
                <span className="text-sm font-medium text-gray-600">
                  {stat.label}
                </span>
              </div>
              <Badge variant="outline" className="text-xs">
                {stat.percentage.toFixed(1)}%
              </Badge>
            </div>
            
            <div className={`text-2xl font-bold ${stat.textColor} mb-2`}>
              {stat.count}
            </div>
            
            <Progress 
              value={stat.percentage} 
              className="h-2"
              style={{
                '--progress-background': stat.color
              } as React.CSSProperties}
            />
            
            <div className="text-xs text-gray-500 mt-1">
              of {inventory.total_units} total units
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
};

export default AvailabilityStats;