import React from 'react';
import { Card, CardContent } from '../ui/card';
import { Badge } from '../ui/badge';
import { Button } from '../ui/button';
import { 
  UnitInventory, 
  UnitStatus, 
  UNIT_STATUS_COLORS 
} from '../../types/unitInventory';

interface UnitBlockProps {
  unit: UnitInventory;
  onClick: (unit: UnitInventory) => void;
  onStatusUpdate: (unitId: number, newStatus: UnitStatus, bookingDetails?: any) => void;
  showActions?: boolean;
}

const UnitBlock: React.FC<UnitBlockProps> = ({
  unit,
  onClick,
  onStatusUpdate,
  showActions = false
}) => {
  const getStatusColor = (status: string): string => {
    return UNIT_STATUS_COLORS[status as UnitStatus] || '#6b7280';
  };

  const formatPrice = (price: number): string => {
    if (price >= 10000000) {
      return `₹${(price / 10000000).toFixed(1)}Cr`;
    } else if (price >= 100000) {
      return `₹${(price / 100000).toFixed(1)}L`;
    } else {
      return `₹${price.toLocaleString()}`;
    }
  };

  const getStatusBadgeVariant = (status: string) => {
    switch (status) {
      case UnitStatus.AVAILABLE:
        return 'default';
      case UnitStatus.BOOKED:
        return 'secondary';
      case UnitStatus.SOLD:
        return 'destructive';
      case UnitStatus.ON_HOLD:
        return 'outline';
      default:
        return 'outline';
    }
  };

  return (
    <Card 
      className={`cursor-pointer transition-all duration-200 hover:shadow-md hover:scale-105 ${
        unit.status === UnitStatus.AVAILABLE ? 'hover:border-green-300' : ''
      }`}
      onClick={() => onClick(unit)}
    >
      <CardContent className="p-3">
        <div className="space-y-2">
          {/* Unit Number and Status Indicator */}
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <div 
                className="w-3 h-3 rounded-full"
                style={{ backgroundColor: getStatusColor(unit.status) }}
                title={unit.status}
              />
              <span className="font-semibold text-sm">{unit.unit_number}</span>
            </div>
            <Badge 
              variant={getStatusBadgeVariant(unit.status)}
              className="text-xs px-1 py-0"
            >
              {unit.status}
            </Badge>
          </div>

          {/* Unit Type */}
          <div className="text-xs text-gray-600">
            {unit.unit_type}
          </div>

          {/* Area */}
          <div className="text-xs text-gray-500">
            {unit.area} sq ft
            {unit.carpet_area && (
              <span className="block">Carpet: {unit.carpet_area} sq ft</span>
            )}
          </div>

          {/* Price */}
          <div className="text-sm font-medium text-gray-900">
            {formatPrice(unit.price)}
          </div>

          {/* Floor Info */}
          <div className="text-xs text-gray-500">
            Floor {unit.floor} • Tower {unit.tower}
          </div>

          {/* Facing */}
          {unit.facing && (
            <div className="text-xs text-gray-500">
              {unit.facing} facing
            </div>
          )}

          {/* Booking Info */}
          {unit.status === UnitStatus.BOOKED && unit.booked_by && (
            <div className="text-xs text-blue-600">
              Booked by: {unit.booked_by}
            </div>
          )}

          {/* Quick Actions */}
          {showActions && unit.status === UnitStatus.AVAILABLE && (
            <div className="flex gap-1 mt-2">
              <Button
                size="sm"
                variant="outline"
                className="text-xs px-2 py-1 h-6"
                onClick={(e) => {
                  e.stopPropagation();
                  onStatusUpdate(unit.id, UnitStatus.BOOKED);
                }}
              >
                Book
              </Button>
              <Button
                size="sm"
                variant="outline"
                className="text-xs px-2 py-1 h-6"
                onClick={(e) => {
                  e.stopPropagation();
                  onStatusUpdate(unit.id, UnitStatus.ON_HOLD);
                }}
              >
                Hold
              </Button>
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
};

export default UnitBlock;