import React, { useState, useEffect } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '../ui/dialog';
import { Button } from '../ui/button';
import { Badge } from '../ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../ui/select';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { Textarea } from '../ui/textarea';
import { Separator } from '../ui/separator';
import { 
  UnitInventory, 
  UnitStatus, 
  UnitDetails,
  UNIT_STATUS_COLORS,
  UNIT_STATUS_DESCRIPTIONS 
} from '../../types/unitInventory';
import { unitInventoryAPI } from '../../api/unitInventory';

interface UnitDetailsModalProps {
  unit: UnitInventory;
  open: boolean;
  onClose: () => void;
  onStatusUpdate: (unitId: number, newStatus: UnitStatus, bookingDetails?: any) => void;
}

const UnitDetailsModal: React.FC<UnitDetailsModalProps> = ({
  unit,
  open,
  onClose,
  onStatusUpdate
}) => {
  const [unitDetails, setUnitDetails] = useState<UnitDetails | null>(null);
  const [loading, setLoading] = useState(false);
  const [showStatusUpdate, setShowStatusUpdate] = useState(false);
  const [newStatus, setNewStatus] = useState<UnitStatus>(unit.status as UnitStatus);
  const [bookingDetails, setBookingDetails] = useState({
    booked_by: '',
    booking_date: '',
    booking_amount: ''
  });

  useEffect(() => {
    if (open && unit.id) {
      loadUnitDetails();
    }
  }, [open, unit.id]);

  const loadUnitDetails = async () => {
    try {
      setLoading(true);
      const details = await unitInventoryAPI.getUnitDetails(unit.id);
      setUnitDetails(details);
    } catch (error) {
      console.error('Error loading unit details:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleStatusUpdate = async () => {
    try {
      const details = newStatus === UnitStatus.BOOKED ? {
        booked_by: bookingDetails.booked_by,
        booking_date: bookingDetails.booking_date,
        booking_amount: bookingDetails.booking_amount ? parseFloat(bookingDetails.booking_amount) : undefined
      } : undefined;

      await onStatusUpdate(unit.id, newStatus, details);
      setShowStatusUpdate(false);
      onClose();
    } catch (error) {
      console.error('Error updating status:', error);
    }
  };

  const formatPrice = (price: number): string => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0
    }).format(price);
  };

  const formatDate = (dateString: string): string => {
    return new Date(dateString).toLocaleDateString('en-IN');
  };

  return (
    <Dialog open={open} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-3">
            <div 
              className="w-4 h-4 rounded-full"
              style={{ backgroundColor: UNIT_STATUS_COLORS[unit.status as UnitStatus] }}
            />
            Unit {unit.unit_number} Details
            <Badge variant="outline">{unit.status}</Badge>
          </DialogTitle>
        </DialogHeader>

        {loading ? (
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          </div>
        ) : (
          <div className="space-y-6">
            {/* Basic Information */}
            <Card>
              <CardHeader>
                <CardTitle>Basic Information</CardTitle>
              </CardHeader>
              <CardContent className="grid grid-cols-2 md:grid-cols-3 gap-4">
                <div>
                  <Label className="text-sm font-medium text-gray-500">Unit Number</Label>
                  <p className="text-lg font-semibold">{unit.unit_number}</p>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Unit Type</Label>
                  <p className="text-lg">{unit.unit_type}</p>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Status</Label>
                  <div className="flex items-center gap-2">
                    <div 
                      className="w-3 h-3 rounded-full"
                      style={{ backgroundColor: UNIT_STATUS_COLORS[unit.status as UnitStatus] }}
                    />
                    <span>{unit.status}</span>
                  </div>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Tower</Label>
                  <p className="text-lg">Tower {unit.tower}</p>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Floor</Label>
                  <p className="text-lg">Floor {unit.floor}</p>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Facing</Label>
                  <p className="text-lg">{unit.facing || 'Not specified'}</p>
                </div>
              </CardContent>
            </Card>

            {/* Area & Pricing */}
            <Card>
              <CardHeader>
                <CardTitle>Area & Pricing</CardTitle>
              </CardHeader>
              <CardContent className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <div>
                  <Label className="text-sm font-medium text-gray-500">Built-up Area</Label>
                  <p className="text-lg font-semibold">{unit.area} sq ft</p>
                </div>
                {unit.carpet_area && (
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Carpet Area</Label>
                    <p className="text-lg">{unit.carpet_area} sq ft</p>
                  </div>
                )}
                {unit.balcony_area && (
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Balcony Area</Label>
                    <p className="text-lg">{unit.balcony_area} sq ft</p>
                  </div>
                )}
                <div>
                  <Label className="text-sm font-medium text-gray-500">Price</Label>
                  <p className="text-lg font-semibold text-green-600">{formatPrice(unit.price)}</p>
                </div>
              </CardContent>
            </Card>

            {/* Amenities */}
            {unit.amenities && unit.amenities.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle>Amenities</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="flex flex-wrap gap-2">
                    {unit.amenities.map((amenity, index) => (
                      <Badge key={index} variant="secondary">
                        {amenity}
                      </Badge>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Booking Information */}
            {(unit.status === UnitStatus.BOOKED || unit.status === UnitStatus.SOLD) && (
              <Card>
                <CardHeader>
                  <CardTitle>Booking Information</CardTitle>
                </CardHeader>
                <CardContent className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  {unit.booked_by && (
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Booked By</Label>
                      <p className="text-lg">{unit.booked_by}</p>
                    </div>
                  )}
                  {unit.booking_date && (
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Booking Date</Label>
                      <p className="text-lg">{formatDate(unit.booking_date)}</p>
                    </div>
                  )}
                  {unit.booking_amount && (
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Booking Amount</Label>
                      <p className="text-lg font-semibold">{formatPrice(unit.booking_amount)}</p>
                    </div>
                  )}
                </CardContent>
              </Card>
            )}

            {/* Project Information */}
            {unitDetails && (
              <Card>
                <CardHeader>
                  <CardTitle>Project Information</CardTitle>
                </CardHeader>
                <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Project Name</Label>
                    <p className="text-lg">{unitDetails.project_name}</p>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Tower Info</Label>
                    <p className="text-lg">
                      {unitDetails.tower_info.total_floors} floors in Tower {unitDetails.tower_info.name}
                    </p>
                  </div>
                </CardContent>
              </Card>
            )}

            <Separator />

            {/* Status Update Section */}
            <div className="flex items-center justify-between">
              <div>
                <h3 className="text-lg font-semibold">Update Unit Status</h3>
                <p className="text-sm text-gray-500">
                  Current status: {UNIT_STATUS_DESCRIPTIONS[unit.status as UnitStatus]}
                </p>
              </div>
              <Button
                variant="outline"
                onClick={() => setShowStatusUpdate(!showStatusUpdate)}
              >
                {showStatusUpdate ? 'Cancel' : 'Update Status'}
              </Button>
            </div>

            {showStatusUpdate && (
              <Card>
                <CardContent className="pt-6 space-y-4">
                  <div>
                    <Label>New Status</Label>
                    <Select value={newStatus} onValueChange={(value) => setNewStatus(value as UnitStatus)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {Object.values(UnitStatus).map(status => (
                          <SelectItem key={status} value={status}>
                            <div className="flex items-center gap-2">
                              <div 
                                className="w-3 h-3 rounded-full"
                                style={{ backgroundColor: UNIT_STATUS_COLORS[status] }}
                              />
                              {status}
                            </div>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  {newStatus === UnitStatus.BOOKED && (
                    <div className="space-y-3">
                      <div>
                        <Label>Booked By</Label>
                        <Input
                          value={bookingDetails.booked_by}
                          onChange={(e) => setBookingDetails(prev => ({
                            ...prev,
                            booked_by: e.target.value
                          }))}
                          placeholder="Customer name"
                        />
                      </div>
                      <div>
                        <Label>Booking Date</Label>
                        <Input
                          type="date"
                          value={bookingDetails.booking_date}
                          onChange={(e) => setBookingDetails(prev => ({
                            ...prev,
                            booking_date: e.target.value
                          }))}
                        />
                      </div>
                      <div>
                        <Label>Booking Amount</Label>
                        <Input
                          type="number"
                          value={bookingDetails.booking_amount}
                          onChange={(e) => setBookingDetails(prev => ({
                            ...prev,
                            booking_amount: e.target.value
                          }))}
                          placeholder="Amount in INR"
                        />
                      </div>
                    </div>
                  )}

                  <div className="flex gap-2">
                    <Button onClick={handleStatusUpdate}>
                      Update Status
                    </Button>
                    <Button variant="outline" onClick={() => setShowStatusUpdate(false)}>
                      Cancel
                    </Button>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>
        )}
      </DialogContent>
    </Dialog>
  );
};

export default UnitDetailsModal;