import { Card } from "../ui/card";
import { useState, useEffect } from "react";
import { getUserVisitSchedules, VisitSchedule } from "../../api/visitSchedules";
import { getUserId } from "../../utils/userUtils";
import { Calendar, Clock, MapPin, Video } from "lucide-react";

export const UpcomingVisits = () => {
  const [visits, setVisits] = useState<VisitSchedule[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchUpcomingVisits();
  }, []);

  const fetchUpcomingVisits = async () => {
    try {
      const userId = getUserId();
      console.log('UpcomingVisits: getUserId returned:', userId);
      if (!userId) {
        console.log('UpcomingVisits: No userId found');
        return;
      }

      console.log('UpcomingVisits: Fetching visits for userId:', userId);
      const response = await getUserVisitSchedules(userId, 1, 10);
      console.log('UpcomingVisits: API response:', response);
      console.log('UpcomingVisits: All visits:', response.visits);
      
      // Filter for upcoming approved visits
      const now = new Date();
      now.setHours(0, 0, 0, 0); // Reset to start of day
      
      const upcomingVisits = response.visits.filter(visit => {
        const visitDate = new Date(visit.preferred_date);
        visitDate.setHours(0, 0, 0, 0); // Reset to start of day
        const isUpcoming = visitDate >= now;
        const isApproved = visit.status === 'seller_approved';
        console.log(`UpcomingVisits: Visit ${visit.id} - date: ${visit.preferred_date}, visitDate: ${visitDate}, now: ${now}, isUpcoming: ${isUpcoming}, isApproved: ${isApproved}`);
        return isUpcoming && isApproved;
      });
      
      console.log('UpcomingVisits: Filtered upcoming visits:', upcomingVisits);
      setVisits(upcomingVisits);
    } catch (error) {
      console.error('UpcomingVisits: Error fetching upcoming visits:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <Card className="p-6">
        <h3 className="text-xl font-semibold mb-4">Upcoming Visits</h3>
        <div className="animate-pulse">Loading...</div>
      </Card>
    );
  }

  return (
    <Card className="p-6">
      <h3 className="text-xl font-semibold mb-4">Upcoming Visits</h3>
      {visits.length === 0 ? (
        <p className="text-gray-600">No upcoming visits scheduled.</p>
      ) : (
        <div className="space-y-3">
          {visits.map((visit) => (
            <div key={visit.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-3">
                <Calendar className="w-4 h-4 text-blue-600" />
                <div>
                  <p className="font-medium text-sm">{visit.property_title}</p>
                  <p className="text-xs text-gray-600">
                    {new Date(visit.preferred_date).toLocaleDateString()} at {visit.preferred_time}
                  </p>
                </div>
              </div>
              <div className="flex items-center gap-2">
                <span className="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full">
                  Approved
                </span>
                {visit.video_call_enabled && visit.video_call_room_id && (
                  <a
                    href={`https://meet.jit.si/${visit.video_call_room_id}`}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="inline-flex items-center px-2 py-1 text-xs font-medium text-white bg-blue-500 rounded hover:bg-blue-600 transition-colors"
                  >
                    <Video className="w-3 h-3 mr-1" />
                    Video
                  </a>
                )}
              </div>
            </div>
          ))}
        </div>
      )}
    </Card>
  );
};
