import React, { useState } from 'react';
import { QueryClient, QueryClientProvider } from '@tanstack/react-query';
import { useAdminStats, useCreditPackages, useCreatePackage, useTogglePackage } from '../../hooks/useWallet';

const queryClient = new QueryClient();

function AdminWalletContent() {
  const { data: packages } = useCreditPackages();
  const { data: stats } = useAdminStats();
  const createMutation = useCreatePackage();
  const toggleMutation = useTogglePackage();
  const [showForm, setShowForm] = useState(false);
  const [formData, setFormData] = useState({
    name: '',
    credits: 0,
    price: 0,
    bonus_credits: 0,
    discount_percent: 0,
    package_type: 'one-time',
    audience_role: 'all'
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      await createMutation.mutateAsync(formData);
      alert('Package created successfully');
      setShowForm(false);
      setFormData({
        name: '',
        credits: 0,
        price: 0,
        bonus_credits: 0,
        discount_percent: 0,
        package_type: 'one-time',
        audience_role: 'all'
      });
    } catch (error) {
      alert('Failed to create package');
    }
  };

  const togglePackageStatus = async (packageId: number) => {
    try {
      await toggleMutation.mutateAsync(packageId);
    } catch (error) {
      alert('Failed to toggle package status');
    }
  };

  return (
    <div className="admin-wallet">
      <h1>Wallet Management</h1>

      {/* Stats Cards */}
      {stats && (
        <div className="stats-grid">
          <div className="stat-card">
            <div className="stat-icon">💰</div>
            <div className="stat-value">₹{stats.total_revenue.toLocaleString()}</div>
            <div className="stat-label">Total Revenue</div>
          </div>
          <div className="stat-card">
            <div className="stat-icon">🪙</div>
            <div className="stat-value">{stats.total_credits_in_circulation.toLocaleString()}</div>
            <div className="stat-label">Credits in Circulation</div>
          </div>
          <div className="stat-card">
            <div className="stat-icon">👥</div>
            <div className="stat-value">{stats.total_users}</div>
            <div className="stat-label">Total Users</div>
          </div>
          <div className="stat-card">
            <div className="stat-icon">📊</div>
            <div className="stat-value">{stats.total_transactions}</div>
            <div className="stat-label">Total Transactions</div>
          </div>
        </div>
      )}

      {/* Packages Section */}
      <div className="packages-section">
        <div className="section-header">
          <h2>Credit Packages</h2>
          <button className="add-button" onClick={() => setShowForm(!showForm)}>
            {showForm ? 'Cancel' : '+ Add Package'}
          </button>
        </div>

        {/* Create Form */}
        {showForm && (
          <form className="package-form" onSubmit={handleSubmit}>
            <div className="form-grid">
              <input
                type="text"
                placeholder="Package Name"
                value={formData.name}
                onChange={(e) => setFormData({...formData, name: e.target.value})}
                required
              />
              <input
                type="number"
                placeholder="Credits"
                value={formData.credits || ''}
                onChange={(e) => setFormData({...formData, credits: parseInt(e.target.value)})}
                required
              />
              <input
                type="number"
                placeholder="Price (₹)"
                value={formData.price || ''}
                onChange={(e) => setFormData({...formData, price: parseFloat(e.target.value)})}
                required
              />
              <input
                type="number"
                placeholder="Bonus Credits"
                value={formData.bonus_credits || ''}
                onChange={(e) => setFormData({...formData, bonus_credits: parseInt(e.target.value)})}
              />
              <input
                type="number"
                placeholder="Discount %"
                value={formData.discount_percent || ''}
                onChange={(e) => setFormData({...formData, discount_percent: parseFloat(e.target.value)})}
              />
              <select
                value={formData.audience_role}
                onChange={(e) => setFormData({...formData, audience_role: e.target.value})}
              >
                <option value="all">All Users</option>
                <option value="owner">Property Owners</option>
                <option value="agent">Agents</option>
                <option value="developer">Developers</option>
                <option value="buyer">Buyers</option>
              </select>
            </div>
            <button type="submit" className="submit-button">Create Package</button>
          </form>
        )}

        {/* Packages Table */}
        <div className="table-container">
          <table className="packages-table">
            <thead>
              <tr>
                <th>Name</th>
                <th>Credits</th>
                <th>Bonus</th>
                <th>Price</th>
                <th>Discount</th>
                <th>Audience</th>
                <th>Status</th>
                <th>Actions</th>
              </tr>
            </thead>
            <tbody>
              {packages && packages.map((pkg) => (
                <tr key={pkg.id}>
                  <td>{pkg.name}</td>
                  <td>{pkg.credits}</td>
                  <td>{pkg.bonus_credits > 0 ? `+${pkg.bonus_credits}` : '-'}</td>
                  <td>₹{pkg.price.toLocaleString()}</td>
                  <td>{pkg.discount_percent > 0 ? `${pkg.discount_percent}%` : '-'}</td>
                  <td>{pkg.audience_role}</td>
                  <td>
                    <span className={`status-badge ${pkg.is_active ? 'active' : 'inactive'}`}>
                      {pkg.is_active ? 'Active' : 'Inactive'}
                    </span>
                  </td>
                  <td>
                    <button
                      className="toggle-button"
                      onClick={() => togglePackageStatus(pkg.id)}
                    >
                      Toggle
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      <style jsx>{`
        .admin-wallet {
          max-width: 1400px;
          margin: 0 auto;
          padding: 20px;
        }

        h1 {
          font-size: 32px;
          margin-bottom: 30px;
          color: #1a1a1a;
        }

        .stats-grid {
          display: grid;
          grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
          gap: 20px;
          margin-bottom: 40px;
        }

        .stat-card {
          background: white;
          border-radius: 12px;
          padding: 25px;
          box-shadow: 0 2px 8px rgba(0,0,0,0.1);
          text-align: center;
        }

        .stat-icon {
          font-size: 40px;
          margin-bottom: 10px;
        }

        .stat-value {
          font-size: 28px;
          font-weight: bold;
          color: #1a1a1a;
          margin-bottom: 5px;
        }

        .stat-label {
          font-size: 14px;
          color: #666;
        }

        .packages-section {
          background: white;
          border-radius: 12px;
          padding: 30px;
          box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .section-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          margin-bottom: 25px;
        }

        h2 {
          font-size: 24px;
          color: #1a1a1a;
        }

        .add-button {
          padding: 10px 20px;
          background: #667eea;
          color: white;
          border: none;
          border-radius: 8px;
          cursor: pointer;
          font-weight: 500;
        }

        .add-button:hover {
          background: #5568d3;
        }

        .package-form {
          background: #f9fafb;
          padding: 25px;
          border-radius: 8px;
          margin-bottom: 25px;
        }

        .form-grid {
          display: grid;
          grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
          gap: 15px;
          margin-bottom: 15px;
        }

        .form-grid input,
        .form-grid select {
          padding: 10px;
          border: 1px solid #e0e0e0;
          border-radius: 6px;
          font-size: 14px;
        }

        .submit-button {
          width: 100%;
          padding: 12px;
          background: #10b981;
          color: white;
          border: none;
          border-radius: 8px;
          cursor: pointer;
          font-weight: 500;
        }

        .submit-button:hover {
          background: #059669;
        }

        .table-container {
          overflow-x: auto;
        }

        .packages-table {
          width: 100%;
          border-collapse: collapse;
        }

        .packages-table th {
          background: #f9fafb;
          padding: 12px;
          text-align: left;
          font-weight: 600;
          color: #666;
          font-size: 14px;
        }

        .packages-table td {
          padding: 15px 12px;
          border-bottom: 1px solid #f0f0f0;
          font-size: 14px;
        }

        .packages-table tr:hover {
          background: #f9fafb;
        }

        .status-badge {
          padding: 4px 12px;
          border-radius: 12px;
          font-size: 12px;
          font-weight: 500;
        }

        .status-badge.active {
          background: #d1fae5;
          color: #065f46;
        }

        .status-badge.inactive {
          background: #fee2e2;
          color: #991b1b;
        }

        .toggle-button {
          padding: 6px 16px;
          background: #f3f4f6;
          border: 1px solid #e0e0e0;
          border-radius: 6px;
          cursor: pointer;
          font-size: 13px;
        }

        .toggle-button:hover {
          background: #e5e7eb;
        }

        @media (max-width: 768px) {
          .stats-grid {
            grid-template-columns: 1fr;
          }

          .section-header {
            flex-direction: column;
            align-items: flex-start;
            gap: 15px;
          }
        }
      `}</style>
    </div>
  );
};

const AdminWallet: React.FC = () => {
  return (
    <QueryClientProvider client={queryClient}>
      <AdminWalletContent />
    </QueryClientProvider>
  );
};

export default AdminWallet;
