import React, { useState } from 'react';
import { useServices, useWalletBalance, useServiceUsage } from '../../hooks/useWallet';

const ServiceCatalog: React.FC = () => {
  const [selectedCategory, setSelectedCategory] = useState<string>('all');
  const userId = 1; // Replace with actual user ID
  const userType = 'owner';

  const { data: services, isLoading } = useServices(selectedCategory === 'all' ? undefined : selectedCategory);
  const { data: balanceData } = useWalletBalance(userId, userType);
  const usageMutation = useServiceUsage(userId, userType);

  const balance = balanceData?.balance || 0;

  const handleUseService = async (serviceCode: string) => {
    try {
      const result = await usageMutation.mutateAsync({ serviceCode });
      alert(result.message);
    } catch (error: any) {
      alert(error.response?.data?.detail || 'Service usage failed');
    }
  };

  const categories = ['all', 'listing', 'feature', 'lead', 'report', 'addon'];
  
  const filteredServices = services || [];

  const getCategoryIcon = (category: string) => {
    const icons: any = {
      listing: '🏠',
      feature: '⭐',
      lead: '👤',
      report: '📊',
      addon: '🎨',
      other: '📦'
    };
    return icons[category] || '📦';
  };

  if (isLoading) return <div className="loading">Loading services...</div>;

  return (
    <div className="service-catalog">
      <div className="header">
        <h2>Service Catalog</h2>
        <div className="balance-badge">
          Balance: <strong>{balance}</strong> credits
        </div>
      </div>

      <div className="category-filter">
        {categories.map(cat => (
          <button
            key={cat}
            className={`filter-btn ${selectedCategory === cat ? 'active' : ''}`}
            onClick={() => setSelectedCategory(cat)}
          >
            {cat.charAt(0).toUpperCase() + cat.slice(1)}
          </button>
        ))}
      </div>

      <div className="services-grid">
        {filteredServices.map((service) => (
          <div key={service.id} className="service-card">
            <div className="service-icon">
              {getCategoryIcon(service.service_category)}
            </div>
            
            <h3>{service.service_name}</h3>
            <p className="service-description">{service.description}</p>
            
            <div className="service-footer">
              <div className="service-cost">
                <span className="cost-amount">{service.credits_required}</span>
                <span className="cost-label">credits</span>
              </div>
              
              <button
                className="use-button"
                onClick={() => handleUseService(service.service_code)}
                disabled={balance < service.credits_required || usageMutation.isPending}
              >
                {usageMutation.isPending ? 'Processing...' : balance < service.credits_required ? 'Insufficient Credits' : 'Use Service'}
              </button>
            </div>
          </div>
        ))}
      </div>

      {filteredServices.length === 0 && (
        <div className="empty-state">
          <p>No services found in this category</p>
        </div>
      )}

      <style jsx>{`
        .service-catalog {
          max-width: 1200px;
          margin: 0 auto;
          padding: 20px;
        }

        .header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          margin-bottom: 30px;
        }

        h2 {
          font-size: 28px;
          color: #1a1a1a;
        }

        .balance-badge {
          background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
          color: white;
          padding: 10px 20px;
          border-radius: 25px;
          font-size: 14px;
        }

        .balance-badge strong {
          font-size: 18px;
          margin: 0 5px;
        }

        .category-filter {
          display: flex;
          gap: 10px;
          margin-bottom: 30px;
          flex-wrap: wrap;
        }

        .filter-btn {
          padding: 8px 20px;
          border: 2px solid #e0e0e0;
          background: white;
          border-radius: 20px;
          cursor: pointer;
          transition: all 0.3s;
          font-size: 14px;
          font-weight: 500;
        }

        .filter-btn:hover {
          border-color: #667eea;
        }

        .filter-btn.active {
          background: #667eea;
          color: white;
          border-color: #667eea;
        }

        .services-grid {
          display: grid;
          grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
          gap: 20px;
        }

        .service-card {
          background: white;
          border: 1px solid #e0e0e0;
          border-radius: 12px;
          padding: 25px;
          transition: all 0.3s;
        }

        .service-card:hover {
          box-shadow: 0 8px 24px rgba(0,0,0,0.1);
          transform: translateY(-2px);
        }

        .service-icon {
          font-size: 40px;
          margin-bottom: 15px;
        }

        .service-card h3 {
          font-size: 18px;
          margin-bottom: 10px;
          color: #1a1a1a;
        }

        .service-description {
          font-size: 14px;
          color: #666;
          margin-bottom: 20px;
          min-height: 40px;
        }

        .service-footer {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding-top: 15px;
          border-top: 1px solid #f0f0f0;
        }

        .service-cost {
          display: flex;
          flex-direction: column;
        }

        .cost-amount {
          font-size: 24px;
          font-weight: bold;
          color: #667eea;
        }

        .cost-label {
          font-size: 12px;
          color: #666;
        }

        .use-button {
          padding: 10px 20px;
          background: #667eea;
          color: white;
          border: none;
          border-radius: 8px;
          cursor: pointer;
          font-size: 14px;
          font-weight: 500;
          transition: background 0.3s;
        }

        .use-button:hover:not(:disabled) {
          background: #5568d3;
        }

        .use-button:disabled {
          background: #ccc;
          cursor: not-allowed;
        }

        .empty-state {
          text-align: center;
          padding: 60px 20px;
          color: #666;
        }

        .loading {
          text-align: center;
          padding: 50px;
          color: #666;
        }

        @media (max-width: 768px) {
          .header {
            flex-direction: column;
            align-items: flex-start;
            gap: 15px;
          }

          .services-grid {
            grid-template-columns: 1fr;
          }
        }
      `}</style>
    </div>
  );
};

export default ServiceCatalog;
