import React from 'react';
import { useTransactionHistory } from '../../hooks/useWallet';

const TransactionHistory: React.FC = () => {
  const userId = 1; // Replace with actual user ID
  const { data: transactions, isLoading } = useTransactionHistory(userId, 50);

  const getTypeColor = (type: string) => {
    switch (type) {
      case 'credit': return '#10b981';
      case 'debit': return '#ef4444';
      case 'refund': return '#3b82f6';
      case 'bonus': return '#f59e0b';
      default: return '#6b7280';
    }
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case 'credit': return '+';
      case 'debit': return '-';
      case 'refund': return '↩';
      case 'bonus': return '🎁';
      default: return '•';
    }
  };

  if (isLoading) return <div className="loading">Loading transactions...</div>;

  return (
    <div className="transaction-history">
      <h2>Transaction History</h2>
      
      {!transactions || transactions.length === 0 ? (
        <div className="empty-state">
          <p>No transactions yet</p>
        </div>
      ) : (
        <div className="transactions-list">
          {transactions.map((txn) => (
            <div key={txn.id} className="transaction-item">
              <div className="txn-icon" style={{ background: getTypeColor(txn.transaction_type) }}>
                {getTypeIcon(txn.transaction_type)}
              </div>
              
              <div className="txn-details">
                <div className="txn-description">{txn.description}</div>
                <div className="txn-date">
                  {new Date(txn.created_at).toLocaleString()}
                </div>
              </div>

              <div className="txn-amount" style={{ color: getTypeColor(txn.transaction_type) }}>
                {txn.transaction_type === 'debit' ? '-' : '+'}{txn.amount}
              </div>

              <div className="txn-balance">
                Balance: {txn.balance_after}
              </div>
            </div>
          ))}
        </div>
      )}

      <style jsx>{`
        .transaction-history {
          max-width: 900px;
          margin: 40px auto;
          padding: 20px;
        }

        h2 {
          font-size: 24px;
          margin-bottom: 20px;
          color: #1a1a1a;
        }

        .empty-state {
          text-align: center;
          padding: 60px 20px;
          color: #666;
        }

        .transactions-list {
          background: white;
          border-radius: 12px;
          overflow: hidden;
          box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .transaction-item {
          display: grid;
          grid-template-columns: 50px 1fr auto auto;
          gap: 15px;
          align-items: center;
          padding: 20px;
          border-bottom: 1px solid #f0f0f0;
          transition: background 0.2s;
        }

        .transaction-item:last-child {
          border-bottom: none;
        }

        .transaction-item:hover {
          background: #f9fafb;
        }

        .txn-icon {
          width: 40px;
          height: 40px;
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
          color: white;
          font-weight: bold;
          font-size: 18px;
        }

        .txn-details {
          flex: 1;
        }

        .txn-description {
          font-size: 15px;
          font-weight: 500;
          color: #1a1a1a;
          margin-bottom: 4px;
        }

        .txn-date {
          font-size: 13px;
          color: #666;
        }

        .txn-amount {
          font-size: 18px;
          font-weight: bold;
          min-width: 80px;
          text-align: right;
        }

        .txn-balance {
          font-size: 14px;
          color: #666;
          min-width: 100px;
          text-align: right;
        }

        .loading {
          text-align: center;
          padding: 50px;
          color: #666;
        }

        @media (max-width: 768px) {
          .transaction-item {
            grid-template-columns: 40px 1fr;
            gap: 10px;
          }

          .txn-amount {
            grid-column: 2;
            text-align: left;
            margin-top: 5px;
          }

          .txn-balance {
            grid-column: 2;
            text-align: left;
          }
        }
      `}</style>
    </div>
  );
};

export default TransactionHistory;
