import React from 'react';
import { useWalletBalance, useCreditPackages, usePurchaseCredits } from '../../hooks/useWallet';

const UserWallet: React.FC = () => {
  const userId = 1; // Replace with actual user ID from auth context
  const userType = 'owner'; // Replace with actual user type

  const { data: balance, isLoading: balanceLoading } = useWalletBalance(userId, userType);
  const { data: packages, isLoading: packagesLoading } = useCreditPackages(userType);
  const purchaseMutation = usePurchaseCredits(userId, userType);

  const handlePurchase = async (packageId: number) => {
    try {
      const result = await purchaseMutation.mutateAsync(packageId);
      alert(result.message);
    } catch (error: any) {
      alert(error.response?.data?.detail || 'Purchase failed');
    }
  };

  if (balanceLoading || packagesLoading) return <div className="loading">Loading wallet...</div>;

  return (
    <div className="wallet-container">
      <h1>My Wallet</h1>

      {/* Balance Card */}
      <div className="balance-card">
        <div className="balance-header">
          <h2>Current Balance</h2>
          <div className="balance-amount">
            <span className="credits">{balance?.balance || 0}</span>
            <span className="label">Credits</span>
          </div>
          <p className="value-estimate">≈ ₹{((balance?.balance || 0) * 10).toLocaleString()}</p>
        </div>
        
        <div className="balance-stats">
          <div className="stat">
            <span className="stat-label">Total Earned</span>
            <span className="stat-value">{balance?.total_earned || 0}</span>
          </div>
          <div className="stat">
            <span className="stat-label">Total Spent</span>
            <span className="stat-value">{balance?.total_spent || 0}</span>
          </div>
          <div className="stat">
            <span className="stat-label">Total Purchased</span>
            <span className="stat-value">{balance?.total_purchased || 0}</span>
          </div>
        </div>
      </div>

      {/* Credit Packages */}
      <div className="packages-section">
        <h2>Buy Credits</h2>
        <div className="packages-grid">
          {packages.map((pkg) => (
            <div key={pkg.id} className={`package-card ${pkg.discount_percent > 0 ? 'popular' : ''}`}>
              {pkg.discount_percent > 0 && (
                <div className="badge">Save {pkg.discount_percent}%</div>
              )}
              
              <h3>{pkg.name}</h3>
              
              <div className="package-credits">
                <span className="main-credits">{pkg.credits}</span>
                {pkg.bonus_credits > 0 && (
                  <span className="bonus">+{pkg.bonus_credits} Bonus</span>
                )}
                <span className="total">= {pkg.credits + pkg.bonus_credits} Total Credits</span>
              </div>

              <div className="package-price">
                <span className="currency">₹</span>
                <span className="amount">{pkg.price.toLocaleString()}</span>
              </div>

              <button
                className="buy-button"
                onClick={() => handlePurchase(pkg.id)}
                disabled={purchaseMutation.isPending}
              >
                {purchaseMutation.isPending ? 'Processing...' : 'Buy Now'}
              </button>
            </div>
          ))}
        </div>
      </div>

      <style jsx>{`
        .wallet-container {
          max-width: 1200px;
          margin: 0 auto;
          padding: 20px;
        }

        h1 {
          font-size: 32px;
          margin-bottom: 30px;
          color: #1a1a1a;
        }

        .balance-card {
          background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
          border-radius: 16px;
          padding: 30px;
          color: white;
          margin-bottom: 40px;
          box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }

        .balance-header {
          text-align: center;
          margin-bottom: 30px;
        }

        .balance-header h2 {
          font-size: 18px;
          opacity: 0.9;
          margin-bottom: 15px;
        }

        .balance-amount {
          display: flex;
          flex-direction: column;
          align-items: center;
        }

        .credits {
          font-size: 56px;
          font-weight: bold;
          line-height: 1;
        }

        .label {
          font-size: 16px;
          opacity: 0.9;
          margin-top: 5px;
        }

        .value-estimate {
          font-size: 14px;
          opacity: 0.8;
          margin-top: 10px;
        }

        .balance-stats {
          display: grid;
          grid-template-columns: repeat(3, 1fr);
          gap: 20px;
          padding-top: 20px;
          border-top: 1px solid rgba(255,255,255,0.2);
        }

        .stat {
          text-align: center;
        }

        .stat-label {
          display: block;
          font-size: 12px;
          opacity: 0.8;
          margin-bottom: 5px;
        }

        .stat-value {
          display: block;
          font-size: 24px;
          font-weight: bold;
        }

        .packages-section h2 {
          font-size: 24px;
          margin-bottom: 20px;
          color: #1a1a1a;
        }

        .packages-grid {
          display: grid;
          grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
          gap: 20px;
        }

        .package-card {
          background: white;
          border: 2px solid #e0e0e0;
          border-radius: 12px;
          padding: 30px;
          text-align: center;
          position: relative;
          transition: all 0.3s ease;
        }

        .package-card:hover {
          transform: translateY(-5px);
          box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }

        .package-card.popular {
          border-color: #667eea;
          box-shadow: 0 5px 20px rgba(102,126,234,0.2);
        }

        .badge {
          position: absolute;
          top: -10px;
          right: 20px;
          background: #ff6b6b;
          color: white;
          padding: 5px 15px;
          border-radius: 20px;
          font-size: 12px;
          font-weight: bold;
        }

        .package-card h3 {
          font-size: 20px;
          margin-bottom: 20px;
          color: #1a1a1a;
        }

        .package-credits {
          margin-bottom: 20px;
        }

        .main-credits {
          display: block;
          font-size: 48px;
          font-weight: bold;
          color: #667eea;
          line-height: 1;
        }

        .bonus {
          display: block;
          color: #ff6b6b;
          font-size: 14px;
          font-weight: bold;
          margin: 5px 0;
        }

        .total {
          display: block;
          font-size: 12px;
          color: #666;
          margin-top: 5px;
        }

        .package-price {
          margin-bottom: 20px;
        }

        .currency {
          font-size: 20px;
          color: #666;
        }

        .amount {
          font-size: 32px;
          font-weight: bold;
          color: #1a1a1a;
        }

        .buy-button {
          width: 100%;
          padding: 12px;
          background: #667eea;
          color: white;
          border: none;
          border-radius: 8px;
          font-size: 16px;
          font-weight: bold;
          cursor: pointer;
          transition: background 0.3s ease;
        }

        .buy-button:hover:not(:disabled) {
          background: #5568d3;
        }

        .buy-button:disabled {
          opacity: 0.6;
          cursor: not-allowed;
        }

        .loading {
          text-align: center;
          padding: 50px;
          font-size: 18px;
          color: #666;
        }

        @media (max-width: 768px) {
          .balance-stats {
            grid-template-columns: 1fr;
            gap: 15px;
          }

          .packages-grid {
            grid-template-columns: 1fr;
          }
        }
      `}</style>
    </div>
  );
};

export default UserWallet;
