import React, { createContext, useContext, useState, useEffect } from 'react';

interface CompareContextType {
  compareList: number[];
  addToCompare: (propertyId: number) => void;
  removeFromCompare: (propertyId: number) => void;
  clearCompare: () => void;
  isInCompare: (propertyId: number) => boolean;
}

const CompareContext = createContext<CompareContextType | undefined>(undefined);

export const CompareProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
  const [compareList, setCompareList] = useState<number[]>(() => {
    const saved = localStorage.getItem('compareList');
    return saved ? JSON.parse(saved) : [];
  });

  useEffect(() => {
    localStorage.setItem('compareList', JSON.stringify(compareList));
  }, [compareList]);

  const addToCompare = (propertyId: number) => {
    if (compareList.length >= 5) {
      alert('You can compare up to 5 properties only');
      return;
    }
    if (!compareList.includes(propertyId)) {
      setCompareList([...compareList, propertyId]);
    }
  };

  const removeFromCompare = (propertyId: number) => {
    setCompareList(compareList.filter(id => id !== propertyId));
  };

  const clearCompare = () => {
    setCompareList([]);
  };

  const isInCompare = (propertyId: number) => {
    return compareList.includes(propertyId);
  };

  return (
    <CompareContext.Provider value={{ compareList, addToCompare, removeFromCompare, clearCompare, isInCompare }}>
      {children}
    </CompareContext.Provider>
  );
};

export const useCompare = () => {
  const context = useContext(CompareContext);
  if (!context) {
    throw new Error('useCompare must be used within CompareProvider');
  }
  return context;
};
