import React, { createContext, useContext, useState, useEffect } from 'react';

const ConfigContext = createContext({});

export function ConfigProvider({ children }) {
  const [config, setConfig] = useState({});
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchConfig();
  }, []);

  const fetchConfig = async () => {
    try {
      const token = localStorage.getItem('access_token');
      if (!token) {
        setLoading(false);
        return;
      }

      const response = await fetch('http://localhost:8090/api/admin/configuration/config/current', {
        headers: { 'Authorization': `Bearer ${token}` }
      });

      if (response.ok) {
        const data = await response.json();
        setConfig(data);
        applyTheme(data);
      }
    } catch (error) {
      console.error('Error fetching config:', error);
    } finally {
      setLoading(false);
    }
  };

  const applyTheme = (config) => {
    if (config.theme_color) {
      document.documentElement.style.setProperty('--primary-color', config.theme_color);
    }
    if (config.logo_url) {
      // Store logo URL for header component
      localStorage.setItem('tenant_logo', config.logo_url);
    }
  };

  const refreshConfig = () => {
    fetchConfig();
  };

  return (
    <ConfigContext.Provider value={{ config, loading, refreshConfig }}>
      {children}
    </ConfigContext.Provider>
  );
}

export function useConfig() {
  const context = useContext(ConfigContext);
  if (!context) {
    throw new Error('useConfig must be used within ConfigProvider');
  }
  return context;
}

// Feature gating hook
export function useFeature(featureName) {
  const { config } = useConfig();
  return config[featureName] === true;
}

// Limit checking hook
export function useLimit(limitName) {
  const { config } = useConfig();
  return config[limitName] || 0;
}
