import React, { createContext, useContext, ReactNode } from 'react';

interface EventBusMessage {
  type: string;
  event_type?: string;
  data?: any;
  timestamp?: string;
}

interface EventBusContextType {
  isConnected: boolean;
  connectionStatus: 'connecting' | 'connected' | 'disconnected' | 'error';
  sendMessage: (message: any) => boolean;
  subscribe: (eventTypes: string[]) => boolean;
  notifications: EventBusMessage[];
  clearNotifications: () => void;
  refreshNotifications: () => void;
}

const EventBusContext = createContext<EventBusContextType | undefined>(undefined);

interface EventBusProviderProps {
  children: ReactNode;
  userId: string;
  token?: string;
  wsUrl?: string;
}

function EventBusProvider({
  children,
}: EventBusProviderProps) {
  // DISABLED WebSocket - Use database polling only
  const contextValue: EventBusContextType = {
    isConnected: false,
    connectionStatus: 'disconnected',
    sendMessage: () => false,
    subscribe: () => false,
    notifications: [],
    clearNotifications: () => {},
    refreshNotifications: () => {}
  };

  return (
    <EventBusContext.Provider value={contextValue}>
      {children}
    </EventBusContext.Provider>
  );
}

function useEventBus(): EventBusContextType {
  const context = useContext(EventBusContext);
  if (!context) {
    throw new Error('useEventBus must be used within an EventBusProvider');
  }
  return context;
}

export { EventBusProvider, useEventBus };