export interface Bank {
  id: string;
  name: string;
  logo: string;
  minRate: number;
  maxRate: number;
  processingFee: string;
  features: string[];
  rateByTenure: {
    [key: number]: number; // tenure in years -> rate
  };
}

// Public Sector Banks (PSU)
const PSU_BANKS: Bank[] = [
  { id: 'sbi', name: 'State Bank of India (SBI)', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=SBI', minRate: 8.5, maxRate: 9.5, processingFee: '0.35% + GST', features: ['Lowest Processing Fee', 'Quick Approval'], rateByTenure: { 5: 8.5, 10: 8.7, 15: 8.9, 20: 9.1, 25: 9.3, 30: 9.5 } },
  { id: 'canara', name: 'Canara Bank', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=CB', minRate: 8.4, maxRate: 9.4, processingFee: '0.35% + GST', features: ['Government Bank', 'Wide Network'], rateByTenure: { 5: 8.4, 10: 8.6, 15: 8.8, 20: 9.0, 25: 9.2, 30: 9.4 } },
  { id: 'bob', name: 'Bank of Baroda', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=BOB', minRate: 8.5, maxRate: 9.5, processingFee: '0.35% + GST', features: ['PSU Bank', 'Low Rates'], rateByTenure: { 5: 8.5, 10: 8.7, 15: 8.9, 20: 9.1, 25: 9.3, 30: 9.5 } },
  { id: 'union', name: 'Union Bank of India', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=UBI', minRate: 8.6, maxRate: 9.6, processingFee: '0.35% + GST', features: ['Government Bank', 'Flexible Terms'], rateByTenure: { 5: 8.6, 10: 8.8, 15: 9.0, 20: 9.2, 25: 9.4, 30: 9.6 } },
  { id: 'indian', name: 'Indian Bank', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=IB', minRate: 8.5, maxRate: 9.5, processingFee: '0.35% + GST', features: ['PSU Bank', 'Quick Processing'], rateByTenure: { 5: 8.5, 10: 8.7, 15: 8.9, 20: 9.1, 25: 9.3, 30: 9.5 } },
  { id: 'pnb', name: 'Punjab National Bank (PNB)', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=PNB', minRate: 8.4, maxRate: 9.4, processingFee: '0.35% + GST', features: ['Government Bank', 'Low Processing Fee'], rateByTenure: { 5: 8.4, 10: 8.6, 15: 8.8, 20: 9.0, 25: 9.2, 30: 9.4 } },
  { id: 'iob', name: 'Indian Overseas Bank', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=IOB', minRate: 8.6, maxRate: 9.6, processingFee: '0.35% + GST', features: ['PSU Bank', 'Wide Network'], rateByTenure: { 5: 8.6, 10: 8.8, 15: 9.0, 20: 9.2, 25: 9.4, 30: 9.6 } },
  { id: 'uco', name: 'UCO Bank', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=UCO', minRate: 8.7, maxRate: 9.7, processingFee: '0.35% + GST', features: ['Government Bank', 'Flexible EMI'], rateByTenure: { 5: 8.7, 10: 8.9, 15: 9.1, 20: 9.3, 25: 9.5, 30: 9.7 } },
  { id: 'boi', name: 'Bank of India', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=BOI', minRate: 8.6, maxRate: 9.6, processingFee: '0.35% + GST', features: ['PSU Bank', 'Low Rates'], rateByTenure: { 5: 8.6, 10: 8.8, 15: 9.0, 20: 9.2, 25: 9.4, 30: 9.6 } },
  { id: 'cbi', name: 'Central Bank of India', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=CBI', minRate: 8.7, maxRate: 9.7, processingFee: '0.35% + GST', features: ['Government Bank', 'Quick Approval'], rateByTenure: { 5: 8.7, 10: 8.9, 15: 9.1, 20: 9.3, 25: 9.5, 30: 9.7 } },
  { id: 'bom', name: 'Bank of Maharashtra', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=BOM', minRate: 8.8, maxRate: 9.8, processingFee: '0.35% + GST', features: ['PSU Bank', 'Regional Focus'], rateByTenure: { 5: 8.8, 10: 9.0, 15: 9.2, 20: 9.4, 25: 9.6, 30: 9.8 } },
  { id: 'psb', name: 'Punjab & Sind Bank', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=PSB', minRate: 8.9, maxRate: 9.9, processingFee: '0.35% + GST', features: ['Government Bank', 'Personal Service'], rateByTenure: { 5: 8.9, 10: 9.1, 15: 9.3, 20: 9.5, 25: 9.7, 30: 9.9 } }
];

// Private Sector Banks
const PRIVATE_BANKS: Bank[] = [
  { id: 'hdfc', name: 'HDFC Bank', logo: 'https://via.placeholder.com/40x40/004c8c/ffffff?text=HDFC', minRate: 8.6, maxRate: 9.8, processingFee: '0.5% + GST', features: ['Digital Process', 'Pre-approved Offers'], rateByTenure: { 5: 8.6, 10: 8.8, 15: 9.0, 20: 9.3, 25: 9.5, 30: 9.8 } },
  { id: 'icici', name: 'ICICI Bank', logo: 'https://via.placeholder.com/40x40/f47920/ffffff?text=ICICI', minRate: 8.7, maxRate: 9.9, processingFee: '0.5% + GST', features: ['Instant Approval', 'Online Application'], rateByTenure: { 5: 8.7, 10: 8.9, 15: 9.1, 20: 9.4, 25: 9.6, 30: 9.9 } },
  { id: 'axis', name: 'Axis Bank', logo: 'https://via.placeholder.com/40x40/97144d/ffffff?text=AXIS', minRate: 8.8, maxRate: 10.0, processingFee: '1% + GST', features: ['Competitive Rates', 'Fast Processing'], rateByTenure: { 5: 8.8, 10: 9.0, 15: 9.2, 20: 9.5, 25: 9.7, 30: 10.0 } },
  { id: 'kotak', name: 'Kotak Mahindra Bank', logo: 'https://via.placeholder.com/40x40/ed1c24/ffffff?text=KMB', minRate: 8.9, maxRate: 10.2, processingFee: '0.5% + GST', features: ['Premium Service', 'Relationship Benefits'], rateByTenure: { 5: 8.9, 10: 9.1, 15: 9.3, 20: 9.6, 25: 9.9, 30: 10.2 } },
  { id: 'idfc', name: 'IDFC First Bank', logo: 'https://via.placeholder.com/40x40/0066cc/ffffff?text=IDFC', minRate: 9.0, maxRate: 10.5, processingFee: '0.5% + GST', features: ['Digital Banking', 'Quick Disbursal'], rateByTenure: { 5: 9.0, 10: 9.2, 15: 9.5, 20: 9.8, 25: 10.1, 30: 10.5 } },
  { id: 'federal', name: 'Federal Bank', logo: 'https://via.placeholder.com/40x40/8B4513/ffffff?text=FB', minRate: 9.1, maxRate: 10.3, processingFee: '0.5% + GST', features: ['South Indian Bank', 'Personal Service'], rateByTenure: { 5: 9.1, 10: 9.3, 15: 9.5, 20: 9.8, 25: 10.0, 30: 10.3 } },
  { id: 'yes', name: 'Yes Bank', logo: 'https://via.placeholder.com/40x40/0066ff/ffffff?text=YES', minRate: 9.2, maxRate: 10.8, processingFee: '0.75% + GST', features: ['Digital Focus', 'Quick Processing'], rateByTenure: { 5: 9.2, 10: 9.5, 15: 9.8, 20: 10.2, 25: 10.5, 30: 10.8 } },
  { id: 'indusind', name: 'IndusInd Bank', logo: 'https://via.placeholder.com/40x40/ff6600/ffffff?text=IIB', minRate: 9.0, maxRate: 10.4, processingFee: '0.5% + GST', features: ['Premium Banking', 'Flexible Terms'], rateByTenure: { 5: 9.0, 10: 9.2, 15: 9.5, 20: 9.8, 25: 10.1, 30: 10.4 } },
  { id: 'rbl', name: 'RBL Bank', logo: 'https://via.placeholder.com/40x40/006600/ffffff?text=RBL', minRate: 9.3, maxRate: 11.0, processingFee: '0.75% + GST', features: ['Retail Focus', 'Quick Approval'], rateByTenure: { 5: 9.3, 10: 9.6, 15: 9.9, 20: 10.3, 25: 10.6, 30: 11.0 } },
  { id: 'sib', name: 'South Indian Bank', logo: 'https://via.placeholder.com/40x40/8B4513/ffffff?text=SIB', minRate: 9.2, maxRate: 10.5, processingFee: '0.5% + GST', features: ['Regional Bank', 'Personal Touch'], rateByTenure: { 5: 9.2, 10: 9.4, 15: 9.7, 20: 10.0, 25: 10.2, 30: 10.5 } },
  { id: 'dcb', name: 'DCB Bank', logo: 'https://via.placeholder.com/40x40/4B0082/ffffff?text=DCB', minRate: 9.5, maxRate: 11.2, processingFee: '0.75% + GST', features: ['Boutique Banking', 'Personalized Service'], rateByTenure: { 5: 9.5, 10: 9.8, 15: 10.1, 20: 10.5, 25: 10.8, 30: 11.2 } },
  { id: 'karnataka', name: 'Karnataka Bank', logo: 'https://via.placeholder.com/40x40/8B0000/ffffff?text=KB', minRate: 9.1, maxRate: 10.3, processingFee: '0.5% + GST', features: ['Regional Focus', 'Local Expertise'], rateByTenure: { 5: 9.1, 10: 9.3, 15: 9.6, 20: 9.9, 25: 10.1, 30: 10.3 } },
  { id: 'cub', name: 'City Union Bank', logo: 'https://via.placeholder.com/40x40/FF4500/ffffff?text=CUB', minRate: 9.3, maxRate: 10.6, processingFee: '0.5% + GST', features: ['South Focus', 'Quick Processing'], rateByTenure: { 5: 9.3, 10: 9.5, 15: 9.8, 20: 10.1, 25: 10.3, 30: 10.6 } },
  { id: 'bandhan', name: 'Bandhan Bank', logo: 'https://via.placeholder.com/40x40/228B22/ffffff?text=BB', minRate: 9.4, maxRate: 10.8, processingFee: '0.75% + GST', features: ['Micro Finance Focus', 'Rural Reach'], rateByTenure: { 5: 9.4, 10: 9.7, 15: 10.0, 20: 10.3, 25: 10.5, 30: 10.8 } },
  { id: 'tmb', name: 'Tamilnad Mercantile Bank', logo: 'https://via.placeholder.com/40x40/8B4513/ffffff?text=TMB', minRate: 9.2, maxRate: 10.4, processingFee: '0.5% + GST', features: ['South Indian Bank', 'SME Focus'], rateByTenure: { 5: 9.2, 10: 9.4, 15: 9.7, 20: 10.0, 25: 10.2, 30: 10.4 } }
];

// Housing Finance Companies
const HFC_COMPANIES: Bank[] = [
  { id: 'lic_hf', name: 'LIC Housing Finance', logo: 'https://via.placeholder.com/40x40/FF6347/ffffff?text=LIC', minRate: 8.5, maxRate: 9.8, processingFee: '0.5% + GST', features: ['LIC Backing', 'Competitive Rates'], rateByTenure: { 5: 8.5, 10: 8.7, 15: 9.0, 20: 9.3, 25: 9.5, 30: 9.8 } },
  { id: 'pnb_hf', name: 'PNB Housing Finance', logo: 'https://via.placeholder.com/40x40/1f4e79/ffffff?text=PNB', minRate: 8.7, maxRate: 10.0, processingFee: '0.5% + GST', features: ['Bank Backing', 'Quick Processing'], rateByTenure: { 5: 8.7, 10: 8.9, 15: 9.2, 20: 9.5, 25: 9.7, 30: 10.0 } },
  { id: 'tata_hf', name: 'Tata Capital Housing Finance', logo: 'https://via.placeholder.com/40x40/0066cc/ffffff?text=TATA', minRate: 8.8, maxRate: 10.2, processingFee: '0.5% + GST', features: ['Tata Group', 'Premium Service'], rateByTenure: { 5: 8.8, 10: 9.0, 15: 9.3, 20: 9.6, 25: 9.9, 30: 10.2 } },
  { id: 'bajaj_hf', name: 'Bajaj Housing Finance', logo: 'https://via.placeholder.com/40x40/FF4500/ffffff?text=BAJAJ', minRate: 8.9, maxRate: 10.5, processingFee: '0.75% + GST', features: ['Bajaj Group', 'Digital Process'], rateByTenure: { 5: 8.9, 10: 9.1, 15: 9.4, 20: 9.8, 25: 10.1, 30: 10.5 } },
  { id: 'indiabulls', name: 'IndiaBulls Housing Finance', logo: 'https://via.placeholder.com/40x40/8B0000/ffffff?text=IB', minRate: 9.0, maxRate: 10.8, processingFee: '0.75% + GST', features: ['Quick Approval', 'Flexible Terms'], rateByTenure: { 5: 9.0, 10: 9.3, 15: 9.6, 20: 10.0, 25: 10.4, 30: 10.8 } },
  { id: 'sundaram', name: 'Sundaram Home Finance', logo: 'https://via.placeholder.com/40x40/4B0082/ffffff?text=SHF', minRate: 9.1, maxRate: 10.6, processingFee: '0.5% + GST', features: ['South Focus', 'Personal Service'], rateByTenure: { 5: 9.1, 10: 9.3, 15: 9.6, 20: 10.0, 25: 10.3, 30: 10.6 } },
  { id: 'aavas', name: 'Aavas Financiers', logo: 'https://via.placeholder.com/40x40/228B22/ffffff?text=AAVAS', minRate: 9.2, maxRate: 11.0, processingFee: '1% + GST', features: ['Affordable Housing', 'Rural Focus'], rateByTenure: { 5: 9.2, 10: 9.5, 15: 9.8, 20: 10.3, 25: 10.6, 30: 11.0 } },
  { id: 'chola', name: 'Cholamandalam Home Finance', logo: 'https://via.placeholder.com/40x40/FF6347/ffffff?text=CHOLA', minRate: 9.3, maxRate: 11.2, processingFee: '1% + GST', features: ['Murugappa Group', 'Quick Disbursal'], rateByTenure: { 5: 9.3, 10: 9.6, 15: 9.9, 20: 10.4, 25: 10.8, 30: 11.2 } },
  { id: 'shriram', name: 'Shriram Housing Finance', logo: 'https://via.placeholder.com/40x40/8B4513/ffffff?text=SHRIRAM', minRate: 9.4, maxRate: 11.5, processingFee: '1% + GST', features: ['Shriram Group', 'Flexible Criteria'], rateByTenure: { 5: 9.4, 10: 9.7, 15: 10.1, 20: 10.6, 25: 11.0, 30: 11.5 } },
  { id: 'homefirst', name: 'Home First Finance Company', logo: 'https://via.placeholder.com/40x40/0066ff/ffffff?text=HFFC', minRate: 9.5, maxRate: 12.0, processingFee: '1.25% + GST', features: ['Affordable Housing', 'Self-Employed Focus'], rateByTenure: { 5: 9.5, 10: 9.8, 15: 10.3, 20: 10.8, 25: 11.4, 30: 12.0 } },
  { id: 'abg_hf', name: 'Aditya Birla Housing Finance', logo: 'https://via.placeholder.com/40x40/FF4500/ffffff?text=ABG', minRate: 9.0, maxRate: 10.5, processingFee: '0.75% + GST', features: ['Birla Group', 'Premium Service'], rateByTenure: { 5: 9.0, 10: 9.2, 15: 9.5, 20: 9.8, 25: 10.1, 30: 10.5 } }
];

// Regional Rural Banks
const RRB_BANKS: Bank[] = [
  { id: 'kgb', name: 'Karnataka Gramin Bank', logo: 'https://via.placeholder.com/40x40/228B22/ffffff?text=KGB', minRate: 8.3, maxRate: 9.2, processingFee: '0.25% + GST', features: ['Rural Focus', 'Government Backed'], rateByTenure: { 5: 8.3, 10: 8.5, 15: 8.7, 20: 8.9, 25: 9.0, 30: 9.2 } },
  { id: 'kvgb', name: 'Karnataka Vikas Grameena Bank', logo: 'https://via.placeholder.com/40x40/228B22/ffffff?text=KVGB', minRate: 8.4, maxRate: 9.3, processingFee: '0.25% + GST', features: ['Rural Development', 'Low Rates'], rateByTenure: { 5: 8.4, 10: 8.6, 15: 8.8, 20: 9.0, 25: 9.1, 30: 9.3 } }
];

// Co-operative Banks
const COOP_BANKS: Bank[] = [
  { id: 'kscab', name: 'Karnataka State Co-Operative Apex Bank', logo: 'https://via.placeholder.com/40x40/8B4513/ffffff?text=KSCAB', minRate: 8.8, maxRate: 9.8, processingFee: '0.5% + GST', features: ['Co-operative', 'Local Focus'], rateByTenure: { 5: 8.8, 10: 9.0, 15: 9.2, 20: 9.4, 25: 9.6, 30: 9.8 } },
  { id: 'citizen', name: 'Citizen Co-operative Bank', logo: 'https://via.placeholder.com/40x40/4B0082/ffffff?text=CCB', minRate: 9.0, maxRate: 10.0, processingFee: '0.5% + GST', features: ['Co-operative', 'Member Benefits'], rateByTenure: { 5: 9.0, 10: 9.2, 15: 9.4, 20: 9.6, 25: 9.8, 30: 10.0 } },
  { id: 'vasavi', name: 'Sri Vasavi Co-operative Bank', logo: 'https://via.placeholder.com/40x40/8B0000/ffffff?text=SVCB', minRate: 9.1, maxRate: 10.1, processingFee: '0.5% + GST', features: ['Regional Co-op', 'Personal Service'], rateByTenure: { 5: 9.1, 10: 9.3, 15: 9.5, 20: 9.7, 25: 9.9, 30: 10.1 } },
  { id: 'janatha', name: 'Janatha Co-operative Bank', logo: 'https://via.placeholder.com/40x40/228B22/ffffff?text=JCB', minRate: 9.2, maxRate: 10.2, processingFee: '0.5% + GST', features: ['Community Bank', 'Local Expertise'], rateByTenure: { 5: 9.2, 10: 9.4, 15: 9.6, 20: 9.8, 25: 10.0, 30: 10.2 } },
  { id: 'tumkur', name: 'The Tumkur Co-operative Bank', logo: 'https://via.placeholder.com/40x40/FF6347/ffffff?text=TCB', minRate: 9.3, maxRate: 10.3, processingFee: '0.5% + GST', features: ['Regional Co-op', 'Agricultural Focus'], rateByTenure: { 5: 9.3, 10: 9.5, 15: 9.7, 20: 9.9, 25: 10.1, 30: 10.3 } },
  { id: 'svc', name: 'Shamrao Vithal Co-operative Bank', logo: 'https://via.placeholder.com/40x40/0066cc/ffffff?text=SVC', minRate: 8.9, maxRate: 9.9, processingFee: '0.5% + GST', features: ['Urban Co-op', 'Digital Banking'], rateByTenure: { 5: 8.9, 10: 9.1, 15: 9.3, 20: 9.5, 25: 9.7, 30: 9.9 } }
];

export const BANK_CATEGORIES = {
  'Public Sector Banks (PSU)': PSU_BANKS,
  'Private Sector Banks': PRIVATE_BANKS,
  'Housing Finance Companies': HFC_COMPANIES,
  'Regional Rural Banks': RRB_BANKS,
  'Co-operative Banks': COOP_BANKS
};

export const BANKS_DATA: Bank[] = [
  ...PSU_BANKS,
  ...PRIVATE_BANKS,
  ...HFC_COMPANIES,
  ...RRB_BANKS,
  ...COOP_BANKS
];

export const getBankById = (id: string): Bank | undefined => {
  return BANKS_DATA.find(bank => bank.id === id);
};

export const getBankRate = (bankId: string, tenure: number): number => {
  const bank = getBankById(bankId);
  if (!bank) return 9.0; // default rate
  
  return bank.rateByTenure[tenure] || bank.minRate;
};