import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { pricingApi } from '../../api/pricing/pricingApi';
import { toast } from 'sonner';

export const usePricingRecommendation = (propertyId: number) => {
  const queryClient = useQueryClient();

  const { data, isLoading, error, refetch } = useQuery({
    queryKey: ['pricing-recommendation', propertyId],
    queryFn: () => pricingApi.getRecommendations(propertyId, 1),
    select: (response) => response.data?.[0] || null
  });

  const generateMutation = useMutation({
    mutationFn: () => pricingApi.generateRecommendation(propertyId),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['pricing-recommendation', propertyId] });
      toast.success('New pricing recommendation generated!');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.detail || 'Failed to generate recommendation');
    }
  });

  const acceptMutation = useMutation({
    mutationFn: (recommendationId: number) => pricingApi.acceptRecommendation(recommendationId),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['pricing-recommendation', propertyId] });
      toast.success('✅ Price updated successfully!');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.detail || 'Failed to accept recommendation');
    }
  });

  const rejectMutation = useMutation({
    mutationFn: ({ recommendationId, reason }: { recommendationId: number; reason: string }) =>
      pricingApi.rejectRecommendation(recommendationId, reason),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['pricing-recommendation', propertyId] });
      toast.success('Recommendation rejected');
    }
  });

  return {
    recommendation: data,
    isLoading,
    error,
    refetch,
    generate: generateMutation.mutate,
    accept: acceptMutation.mutate,
    reject: rejectMutation.mutate,
    isGenerating: generateMutation.isPending,
    isAccepting: acceptMutation.isPending,
    isRejecting: rejectMutation.isPending
  };
};

export const useBuyerInsights = (propertyId: number) => {
  return useQuery({
    queryKey: ['buyer-insights', propertyId],
    queryFn: () => pricingApi.getBuyerInsights(propertyId),
    select: (response) => response.data
  });
};

export const useSellerRecommendations = (sellerId: number) => {
  return useQuery({
    queryKey: ['seller-recommendations', sellerId],
    queryFn: () => pricingApi.getSellerRecommendations(sellerId),
    select: (response) => response.data
  });
};
