import { useState, useEffect, useCallback, useRef } from 'react';
import { auctionAPI, userAuctionAPI, AuctionSession, Bid, AuctionWebSocket, auctionUtils } from '../api/auctions';

interface AuctionState {
  session: AuctionSession | null;
  bids: Bid[];
  currentBid: number;
  timeRemaining: string;
  isRegistered: boolean;
  registrationStatus: any | null;
  isLoading: boolean;
  error: string | null;
  winnerName: string | null;
}

export const useAuction = (sessionId: string) => {
  const [state, setState] = useState<AuctionState>({
    session: null,
    bids: [],
    currentBid: 0,
    timeRemaining: '',
    isRegistered: false,
    registrationStatus: null,
    isLoading: true,
    error: null,
    winnerName: null
  });

  const wsRef = useRef<AuctionWebSocket | null>(null);
  const intervalRef = useRef<NodeJS.Timeout | null>(null);
  const refreshIntervalRef = useRef<NodeJS.Timeout | null>(null);

  // Fetch auction status with winner info
  const fetchAuctionDetails = useCallback(async () => {
    if (!sessionId) return;
    
    try {
      setState(prev => ({ ...prev, isLoading: true, error: null }));
      
      // Get current status (works with property_id or session_id)
      const status = await auctionAPI.getCurrentStatus(sessionId);
      
      if (status && status.session) {
        setState(prev => ({
          ...prev,
          session: status.session,
          currentBid: status.current_highest_bid?.bid_amount || status.session.current_highest_bid || status.session.starting_bid || 0,
          isRegistered: status.is_user_registered || false,
          registrationStatus: status.user_registration_status ? { registration_status: status.user_registration_status, registration_fee_paid: true } : null,
          winnerName: status.winner_name || null,
          isLoading: false
        }));
      } else {
        setState(prev => ({ ...prev, session: null, isLoading: false }));
      }
    } catch (error) {
      console.warn('Failed to fetch auction details:', error);
      setState(prev => ({ ...prev, session: null, isLoading: false }));
    }
  }, [sessionId]);

  // Update time remaining
  const updateTimeRemaining = useCallback(() => {
    if (state.session?.end_time) {
      const timeData = auctionUtils.getTimeRemaining(state.session.end_time);
      setState(prev => ({ ...prev, timeRemaining: timeData.display }));
      
      if (timeData.expired && state.session.status === 'active') {
        // Refresh to get updated status
        fetchAuctionDetails();
      }
    }
  }, [state.session?.end_time, state.session?.status, fetchAuctionDetails]);

  // Place bid
  const placeBid = useCallback(async (amount: number) => {
    try {
      setState(prev => ({ ...prev, isLoading: true, error: null }));
      const result = await userAuctionAPI.placeBid(sessionId, amount);
      await fetchAuctionDetails();
      return result;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Failed to place bid';
      setState(prev => ({ ...prev, error: errorMessage, isLoading: false }));
      throw error;
    }
  }, [sessionId, fetchAuctionDetails]);

  const registerForAuction = useCallback(async () => {
    try {
      setState(prev => ({ ...prev, isLoading: true, error: null }));
      const result = await userAuctionAPI.registerForAuction(sessionId);
      
      if (result && (result.success !== false)) {
        setState(prev => ({ ...prev, isRegistered: true, registrationStatus: result, isLoading: false }));
      } else {
        setState(prev => ({ ...prev, error: result?.message || 'Failed to register', isLoading: false }));
      }
      return result;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Failed to register';
      setState(prev => ({ ...prev, error: errorMessage, isLoading: false }));
      throw error;
    }
  }, [sessionId]);

  // Set up time remaining interval
  useEffect(() => {
    if (state.session?.end_time && state.session.status === 'active') {
      updateTimeRemaining();
      intervalRef.current = setInterval(updateTimeRemaining, 1000);
    }

    return () => {
      if (intervalRef.current) {
        clearInterval(intervalRef.current);
        intervalRef.current = null;
      }
    };
  }, [state.session?.end_time, state.session?.status, updateTimeRemaining]);

  // Auto-refresh for active auctions
  useEffect(() => {
    if (state.session?.status === 'active') {
      refreshIntervalRef.current = setInterval(() => {
        fetchAuctionDetails();
      }, 10000); // Refresh every 10 seconds
    }

    return () => {
      if (refreshIntervalRef.current) {
        clearInterval(refreshIntervalRef.current);
        refreshIntervalRef.current = null;
      }
    };
  }, [state.session?.status, fetchAuctionDetails]);

  // Initial data fetch
  useEffect(() => {
    if (sessionId) {
      fetchAuctionDetails();
    }
  }, [sessionId, fetchAuctionDetails]);

  return {
    ...state,
    placeBid,
    registerForAuction,
    refreshData: fetchAuctionDetails,
    validateBid: (amount: number) => 
      auctionUtils.validateBid(amount, state.currentBid, state.session?.minimum_increment || 0)
  };
};

// Hook for managing multiple auctions (for auction list pages)
export const useAuctions = (type: 'active' | 'upcoming' = 'active') => {
  const [auctions, setAuctions] = useState<AuctionSession[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchAuctions = useCallback(async () => {
    try {
      setIsLoading(true);
      setError(null);
      
      const result = type === 'active' 
        ? await auctionAPI.getActiveAuctions()
        : await auctionAPI.getUpcomingAuctions();
      
      setAuctions(result.auctions || []);
    } catch (error) {
      setError(error instanceof Error ? error.message : 'Failed to fetch auctions');
    } finally {
      setIsLoading(false);
    }
  }, [type]);

  useEffect(() => {
    fetchAuctions();
  }, [fetchAuctions]);

  return {
    auctions,
    isLoading,
    error,
    refreshAuctions: fetchAuctions
  };
};