import { useCallback } from 'react';
import { auditService } from '../services/auditService';

export interface UseAuditLogReturn {
  logAuthEvent: (
    operation: 'LOGIN_ATTEMPT' | 'LOGIN_SUCCESS' | 'LOGIN_FAILURE' | 'LOGOUT' | 'SESSION_EXPIRED',
    eventData?: Record<string, any>
  ) => Promise<void>;
  
  logDocumentEvent: (
    operation: 'DOCUMENT_VIEW' | 'DOCUMENT_DOWNLOAD' | 'DOCUMENT_UPLOAD_ATTEMPT',
    documentId: string,
    eventData?: Record<string, any>
  ) => Promise<void>;
  
  logPropertyEvent: (
    operation: 'PROPERTY_VIEW' | 'PROPERTY_INQUIRY' | 'PROPERTY_SAVE',
    propertyId: string,
    eventData?: Record<string, any>
  ) => Promise<void>;
  
  logPaymentEvent: (
    operation: 'PAYMENT_INITIATED' | 'PAYMENT_COMPLETED' | 'PAYMENT_FAILED',
    paymentId?: string,
    eventData?: Record<string, any>
  ) => Promise<void>;
  
  logDataExportEvent: (
    operation: 'DATA_EXPORT_REQUEST' | 'DATA_EXPORT_DOWNLOAD',
    exportType: string,
    eventData?: Record<string, any>
  ) => Promise<void>;
  
  logSecurityEvent: (
    operation: string,
    resourceType: string,
    resourceId?: string,
    eventData?: Record<string, any>
  ) => Promise<void>;
}

/**
 * Custom hook for audit logging
 * Provides easy-to-use functions for logging various types of events
 */
export const useAuditLog = (): UseAuditLogReturn => {
  const logAuthEvent = useCallback(async (
    operation: 'LOGIN_ATTEMPT' | 'LOGIN_SUCCESS' | 'LOGIN_FAILURE' | 'LOGOUT' | 'SESSION_EXPIRED',
    eventData?: Record<string, any>
  ) => {
    await auditService.logAuthEvent(operation, eventData);
  }, []);

  const logDocumentEvent = useCallback(async (
    operation: 'DOCUMENT_VIEW' | 'DOCUMENT_DOWNLOAD' | 'DOCUMENT_UPLOAD_ATTEMPT',
    documentId: string,
    eventData?: Record<string, any>
  ) => {
    await auditService.logDocumentEvent(operation, documentId, eventData);
  }, []);

  const logPropertyEvent = useCallback(async (
    operation: 'PROPERTY_VIEW' | 'PROPERTY_INQUIRY' | 'PROPERTY_SAVE',
    propertyId: string,
    eventData?: Record<string, any>
  ) => {
    await auditService.logPropertyEvent(operation, propertyId, eventData);
  }, []);

  const logPaymentEvent = useCallback(async (
    operation: 'PAYMENT_INITIATED' | 'PAYMENT_COMPLETED' | 'PAYMENT_FAILED',
    paymentId?: string,
    eventData?: Record<string, any>
  ) => {
    await auditService.logPaymentEvent(operation, paymentId, eventData);
  }, []);

  const logDataExportEvent = useCallback(async (
    operation: 'DATA_EXPORT_REQUEST' | 'DATA_EXPORT_DOWNLOAD',
    exportType: string,
    eventData?: Record<string, any>
  ) => {
    await auditService.logDataExportEvent(operation, exportType, eventData);
  }, []);

  const logSecurityEvent = useCallback(async (
    operation: string,
    resourceType: string,
    resourceId?: string,
    eventData?: Record<string, any>
  ) => {
    await auditService.logSecurityEvent(operation, resourceType, resourceId, eventData);
  }, []);

  return {
    logAuthEvent,
    logDocumentEvent,
    logPropertyEvent,
    logPaymentEvent,
    logDataExportEvent,
    logSecurityEvent
  };
};