import { useState, useCallback, useEffect } from 'react';
import { 
  uploadPropertyImages, 
  uploadPropertyVideos, 
  uploadPropertyDocuments,
  getPropertyMedia,
  generateDownloadURL,
  replaceMedia,
  deleteMedia,
  getMediaMetadata,
  MediaUploadResponse 
} from '../api/mediaUpload';
import { toast } from 'sonner';

export interface MediaFile {
  id: string;
  url: string;
  type: 'image' | 'video' | 'document';
  category?: string;
  media_type?: 'image' | 'video' | 'document';
  is_primary?: boolean;
  file_name?: string;
  file_size?: number;
  document_type?: string;
  thumbnail_url?: string;
  title?: string;
}

export interface MediaState {
  images: MediaFile[];
  videos: MediaFile[];
  documents: MediaFile[];
  loading: boolean;
  uploading: boolean;
  error: string | null;
}

export const useAzureBlobMedia = (propertyId?: string) => {
  const [mediaState, setMediaState] = useState<MediaState>({
    images: [],
    videos: [],
    documents: [],
    loading: false,
    uploading: false,
    error: null,
  });

  // Load all media for a property
  const loadPropertyMedia = useCallback(async (propId: string) => {
    if (!propId) return;
    
    console.log('🔍 Loading media for property:', propId);
    setMediaState(prev => ({ ...prev, loading: true, error: null }));
    
    try {
      const mediaData = await getPropertyMedia(propId);
      console.log('📦 Media data received:', {
        propertyId: propId,
        mediaKeys: Object.keys(mediaData),
        imagesCount: mediaData.images?.length || 0,
        videosCount: mediaData.videos?.length || 0,
        documentsCount: mediaData.documents?.length || 0,
        mediaArrayCount: mediaData.media?.length || 0,
        fullResponse: mediaData
      });
      
      // Handle different response structures
      let images = [];
      let videos = [];
      let documents = [];
      
      if (mediaData.media && Array.isArray(mediaData.media)) {
        // New structure: media array with type field
        const baseUrl = 'http://localhost:8090';
        
        images = mediaData.media.filter(m => m.type === 'image').map(m => {
          let url = m.file_url;
          // Convert relative URLs to absolute URLs
          if (url && url.startsWith('/')) {
            url = baseUrl + url;
          }
          console.log('🖼️ Processing image:', { id: m.id, originalUrl: m.file_url, processedUrl: url });
          return {
            id: m.id.toString(),
            url: url,
            type: 'image' as const,
            is_primary: m.is_primary,
            file_name: m.file_name
          };
        });
        
        videos = mediaData.media.filter(m => m.type === 'video').map(m => {
          let url = m.file_url;
          // Convert relative URLs to absolute URLs
          if (url && url.startsWith('/')) {
            url = baseUrl + url;
          }
          console.log('🎥 Processing video:', { id: m.id, originalUrl: m.file_url, processedUrl: url });
          return {
            id: m.id.toString(),
            url: url,
            type: 'video' as const,
            file_name: m.file_name
          };
        });
        
        documents = mediaData.media.filter(m => m.type === 'document').map(m => {
          let url = m.file_url;
          // Convert relative URLs to absolute URLs
          if (url && url.startsWith('/')) {
            url = baseUrl + url;
          }
          return {
            id: m.id.toString(),
            url: url,
            type: 'document' as const,
            file_name: m.file_name
          };
        });
      } else {
        // Legacy structure
        images = mediaData.images || [];
        videos = mediaData.videos || [];
        documents = mediaData.documents || [];
      }
      
      console.log('✅ Processed media:', {
        imagesCount: images.length,
        videosCount: videos.length,
        documentsCount: documents.length,
        imageUrls: images.map(img => img.url)
      });
      
      // Skip download URL generation for now to test direct URLs
      console.log('⏭️ Skipping download URL generation, using direct URLs');
      
      setMediaState(prev => ({
        ...prev,
        images,
        videos,
        documents,
        loading: false,
      }));
    } catch (error) {
      console.error('💥 Error loading media:', error);
      const errorMessage = error instanceof Error ? error.message : 'Failed to load media';
      setMediaState(prev => ({ 
        ...prev, 
        loading: false, 
        error: errorMessage 
      }));
      // Don't show toast error for now to avoid spam
    }
  }, []);

  // Upload images with category
  const uploadImages = useCallback(async (propId: string, files: File[], category: string = 'general') => {
    if (!propId || !files.length) return { success: false, error: 'Invalid parameters' };
    
    setMediaState(prev => ({ ...prev, uploading: true, error: null }));
    
    try {
      const result = await uploadPropertyImages(propId, files, category);
      
      if (result.success && result.images) {
        setMediaState(prev => ({
          ...prev,
          images: [...prev.images, ...result.images!],
          uploading: false,
        }));
        
        toast.success(`${files.length} ${category} image(s) uploaded successfully`);
        return { success: true, data: result.images };
      } else {
        throw new Error(result.error || 'Upload failed');
      }
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Upload failed';
      setMediaState(prev => ({ 
        ...prev, 
        uploading: false, 
        error: errorMessage 
      }));
      toast.error(`${category} image upload failed: ${errorMessage}`);
      return { success: false, error: errorMessage };
    }
  }, []);

  // Upload videos with category
  const uploadVideos = useCallback(async (propId: string, files: File[], category: string = 'general') => {
    if (!propId || !files.length) return { success: false, error: 'Invalid parameters' };
    
    setMediaState(prev => ({ ...prev, uploading: true, error: null }));
    
    try {
      const result = await uploadPropertyVideos(propId, files, category);
      
      if (result.success && result.videos) {
        setMediaState(prev => ({
          ...prev,
          videos: [...prev.videos, ...result.videos!],
          uploading: false,
        }));
        
        toast.success(`${files.length} ${category} video(s) uploaded successfully`);
        return { success: true, data: result.videos };
      } else {
        throw new Error(result.error || 'Upload failed');
      }
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Upload failed';
      setMediaState(prev => ({ 
        ...prev, 
        uploading: false, 
        error: errorMessage 
      }));
      toast.error(`${category} video upload failed: ${errorMessage}`);
      return { success: false, error: errorMessage };
    }
  }, []);

  // Upload documents
  const uploadDocuments = useCallback(async (propId: string, files: File[], documentTypes: string[]) => {
    if (!propId || !files.length) return { success: false, error: 'Invalid parameters' };
    
    setMediaState(prev => ({ ...prev, uploading: true, error: null }));
    
    try {
      const result = await uploadPropertyDocuments(propId, files, documentTypes);
      
      if (result.success && result.documents) {
        setMediaState(prev => ({
          ...prev,
          documents: [...prev.documents, ...result.documents!],
          uploading: false,
        }));
        
        toast.success(`${files.length} document(s) uploaded successfully`);
        return { success: true, data: result.documents };
      } else {
        throw new Error(result.error || 'Upload failed');
      }
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Upload failed';
      setMediaState(prev => ({ 
        ...prev, 
        uploading: false, 
        error: errorMessage 
      }));
      toast.error(`Document upload failed: ${errorMessage}`);
      return { success: false, error: errorMessage };
    }
  }, []);

  // Generate download URL for secure access
  const getDownloadUrl = useCallback(async (mediaId: string) => {
    try {
      const result = await generateDownloadURL(mediaId);
      
      if (result.success !== false && result.download_url) {
        return { success: true, url: result.download_url };
      } else {
        throw new Error(result.error || 'Failed to generate download URL');
      }
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Failed to generate URL';
      toast.error(`Download failed: ${errorMessage}`);
      return { success: false, error: errorMessage };
    }
  }, []);

  // Replace existing media file
  const replaceMediaFile = useCallback(async (mediaId: string, newFile: File, mediaType: 'image' | 'video' | 'document') => {
    setMediaState(prev => ({ ...prev, uploading: true, error: null }));
    
    try {
      const result = await replaceMedia(mediaId, newFile);
      
      if (result.success) {
        // Update the specific media item in state
        setMediaState(prev => {
          const updateMediaArray = (items: MediaFile[]) => 
            items.map(item => 
              item.id === mediaId 
                ? { ...item, ...result, id: mediaId }
                : item
            );

          return {
            ...prev,
            images: mediaType === 'image' ? updateMediaArray(prev.images) : prev.images,
            videos: mediaType === 'video' ? updateMediaArray(prev.videos) : prev.videos,
            documents: mediaType === 'document' ? updateMediaArray(prev.documents) : prev.documents,
            uploading: false,
          };
        });
        
        toast.success('Media file replaced successfully');
        return { success: true };
      } else {
        throw new Error(result.error || 'Replace failed');
      }
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Replace failed';
      setMediaState(prev => ({ 
        ...prev, 
        uploading: false, 
        error: errorMessage 
      }));
      toast.error(`Replace failed: ${errorMessage}`);
      return { success: false, error: errorMessage };
    }
  }, []);

  // Delete media file
  const deleteMediaFile = useCallback(async (mediaId: string, mediaType: 'image' | 'video' | 'document') => {
    try {
      const result = await deleteMedia(mediaId);
      
      if (result.success) {
        // Remove the media item from state
        setMediaState(prev => ({
          ...prev,
          images: mediaType === 'image' ? prev.images.filter(item => item.id !== mediaId) : prev.images,
          videos: mediaType === 'video' ? prev.videos.filter(item => item.id !== mediaId) : prev.videos,
          documents: mediaType === 'document' ? prev.documents.filter(item => item.id !== mediaId) : prev.documents,
        }));
        
        toast.success('Media file deleted successfully');
        return { success: true };
      } else {
        throw new Error(result.error || 'Delete failed');
      }
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Delete failed';
      toast.error(`Delete failed: ${errorMessage}`);
      return { success: false, error: errorMessage };
    }
  }, []);

  // Get media metadata
  const getMetadata = useCallback(async (mediaId: string) => {
    try {
      const result = await getMediaMetadata(mediaId);
      return result;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Failed to fetch metadata';
      toast.error(`Metadata fetch failed: ${errorMessage}`);
      return { success: false, error: errorMessage };
    }
  }, []);

  // Auto-load media when propertyId changes
  useEffect(() => {
    if (propertyId) {
      loadPropertyMedia(propertyId);
    }
  }, [propertyId, loadPropertyMedia]);

  return {
    // State
    ...mediaState,
    
    // Actions
    loadPropertyMedia,
    uploadImages,
    uploadVideos,
    uploadDocuments,
    getDownloadUrl,
    replaceMediaFile,
    deleteMediaFile,
    getMetadata,
    
    // Computed values
    totalMediaCount: mediaState.images.length + mediaState.videos.length + mediaState.documents.length,
    hasMedia: mediaState.images.length > 0 || mediaState.videos.length > 0 || mediaState.documents.length > 0,
    
    // Category-specific getters
    getDroneImages: () => mediaState.images.filter(img => img.category === 'drone'),
    getDroneVideos: () => mediaState.videos.filter(video => video.category === 'drone'),
    get3DImages: () => mediaState.images.filter(img => img.category === '3d'),
    get3DVideos: () => mediaState.videos.filter(video => video.category === '3d'),
    getGeneralImages: () => mediaState.images.filter(img => !img.category || img.category === 'general'),
    getGeneralVideos: () => mediaState.videos.filter(video => !video.category || video.category === 'general'),
  };
};

export default useAzureBlobMedia;