import { useState, useEffect } from 'react';
import { loanCalculatorAPI, BankRateResponse } from '../api/loanCalculator';
import { BANKS_DATA } from '../data/banks';

export const useBankRates = (state?: string, city?: string) => {
  const [bankRates, setBankRates] = useState(BANKS_DATA);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [lastUpdated, setLastUpdated] = useState<string | null>(null);

  const fetchBankRates = async () => {
    setLoading(true);
    setError(null);
    
    try {
      const response = await loanCalculatorAPI.getBankRates({ state, city });
      
      if (response.success) {
        // Transform API data to match frontend format
        const transformedBanks = response.data.banks.map(bank => ({
          id: bank.bankId,
          name: bank.bankName,
          logo: bank.logo,
          minRate: bank.rates.minRate,
          maxRate: bank.rates.maxRate,
          processingFee: bank.processingFee.type === 'percentage' 
            ? `${bank.processingFee.value}%${bank.processingFee.gst ? ' + GST' : ''}`
            : `₹${bank.processingFee.value.toLocaleString('en-IN')}${bank.processingFee.gst ? ' + GST' : ''}`,
          features: bank.features,
          rateByTenure: bank.rates.rateByTenure
        }));
        
        setBankRates(transformedBanks);
        setLastUpdated(response.data.lastUpdated);
      }
    } catch (err) {
      console.warn('Failed to fetch bank rates, using fallback data:', err);
      setError('Using offline bank data');
      // Keep using BANKS_DATA as fallback
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchBankRates();
  }, [state, city]);

  return {
    bankRates,
    loading,
    error,
    lastUpdated,
    refetch: fetchBankRates
  };
};