
import { useState, useEffect, useCallback } from 'react';
import { adminService } from '../api/adminService';

// Interfaces based on backend response from ConnectorService.get_active_connectors
export interface Connector {
    id: number;
    provider_slug: string;
    provider_name: string;
    status: 'active' | 'pending_oauth' | 'inactive';
    last_synced_at?: string;
    created_at: string;
    logo_url?: string;
    auth_type: 'oauth2' | 'api_key';
    // Legacy fields for compatibility
    integration_type?: string;
    is_connected?: boolean;
    name?: string;
}

export interface SyncJob {
    id: number;
    job_type: string;
    source_type: string;
    target_type: string;
    status: 'pending' | 'running' | 'completed' | 'failed';
    created_at: string;
    progress_percentage: number;
    message?: string;
    processed_records?: number;
    total_records?: number;
    duration?: number;
}

export const useConnectorAPI = () => {
    const [connectors, setConnectors] = useState<Connector[]>([]);
    const [syncJobs, setSyncJobs] = useState<SyncJob[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);

    const fetchConnectors = useCallback(async () => {
        try {
            setLoading(true);
            const res = await adminService.connectors.getConnectors();
            setConnectors(res.data);
            setError(null);
        } catch (err: any) {
            setError(err.message || 'Failed to fetch connectors');
        } finally {
            setLoading(false);
        }
    }, []);

    const fetchSyncJobs = useCallback(async (connectorId?: number | null) => {
        try {
            const res = await adminService.connectors.getSyncJobs(connectorId);
            setSyncJobs(res.data);
        } catch (err: any) {
            console.error(err);
        }
    }, []);

    useEffect(() => {
        fetchConnectors();
        fetchSyncJobs();
    }, [fetchConnectors, fetchSyncJobs]);

    return {
        connectors,
        syncJobs,
        loading,
        error,
        refetch: fetchConnectors,
        fetchSyncJobs
    };
};
