import { useState, useEffect, useCallback } from 'react';
import DataRetentionAPI from '../api/dataRetention';
import {
  DataRetentionPolicy,
  ErasureRequest,
  DataExport,
  UserDataSummary,
  RetentionStats,
  PaginatedResponse
} from '../types/dataRetention';

// Hook for retention policies
export const useRetentionPolicies = (activeOnly = false) => {
  const [policies, setPolicies] = useState<DataRetentionPolicy[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchPolicies = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      const data = await DataRetentionAPI.getRetentionPolicies(activeOnly);
      setPolicies(data);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch policies');
    } finally {
      setLoading(false);
    }
  }, [activeOnly]);

  const createPolicy = async (policy: Partial<DataRetentionPolicy>) => {
    try {
      const newPolicy = await DataRetentionAPI.createRetentionPolicy(policy);
      setPolicies(prev => [newPolicy, ...prev]);
      return newPolicy;
    } catch (err) {
      throw new Error(err instanceof Error ? err.message : 'Failed to create policy');
    }
  };

  const updatePolicy = async (id: number, updates: Partial<DataRetentionPolicy>) => {
    try {
      const updatedPolicy = await DataRetentionAPI.updateRetentionPolicy(id, updates);
      setPolicies(prev => prev.map(p => p.id === id ? updatedPolicy : p));
      return updatedPolicy;
    } catch (err) {
      throw new Error(err instanceof Error ? err.message : 'Failed to update policy');
    }
  };

  useEffect(() => {
    fetchPolicies();
  }, [fetchPolicies]);

  return {
    policies,
    loading,
    error,
    refetch: fetchPolicies,
    createPolicy,
    updatePolicy
  };
};

// Hook for erasure requests
export const useErasureRequests = (filters: { status?: string; page?: number; page_size?: number } = {}) => {
  const [requests, setRequests] = useState<ErasureRequest[]>([]);
  const [pagination, setPagination] = useState({
    total_count: 0,
    page: 1,
    page_size: 20,
    has_next: false,
    has_previous: false
  });
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchRequests = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      const response = await DataRetentionAPI.getErasureRequests(filters);
      setRequests(response.data);
      setPagination({
        total_count: response.total_count,
        page: response.page,
        page_size: response.page_size,
        has_next: response.has_next,
        has_previous: response.has_previous
      });
    } catch (err) {
      console.error('Error fetching erasure requests:', err);
      setError(err instanceof Error ? err.message : 'Failed to fetch erasure requests');
      setRequests([]);
      setPagination({
        total_count: 0,
        page: 1,
        page_size: 20,
        has_next: false,
        has_previous: false
      });
    } finally {
      setLoading(false);
    }
  }, [JSON.stringify(filters)]);

  const createRequest = async (request: {
    request_type: 'partial' | 'complete';
    reason: string;
    data_categories?: string[];
  }) => {
    try {
      const newRequest = await DataRetentionAPI.createErasureRequest(request);
      await fetchRequests(); // Refresh the list
      return newRequest;
    } catch (err) {
      throw new Error(err instanceof Error ? err.message : 'Failed to create erasure request');
    }
  };

  const updateRequest = async (id: number, updates: {
    status?: 'pending' | 'approved' | 'rejected' | 'completed';
    notes?: string;
  }) => {
    try {
      const updatedRequest = await DataRetentionAPI.updateErasureRequest(id, updates);
      setRequests(prev => prev.map(r => r.id === id ? updatedRequest : r));
      return updatedRequest;
    } catch (err) {
      throw new Error(err instanceof Error ? err.message : 'Failed to update erasure request');
    }
  };

  useEffect(() => {
    fetchRequests();
  }, [fetchRequests]);

  return {
    requests,
    pagination,
    loading,
    error,
    refetch: fetchRequests,
    createRequest,
    updateRequest
  };
};

// Hook for data exports
export const useDataExports = () => {
  const [exports, setExports] = useState<DataExport[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchExports = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      const data = await DataRetentionAPI.getDataExports();
      setExports(data);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch data exports');
    } finally {
      setLoading(false);
    }
  }, []);

  const requestExport = async (exportType: 'personal_data' | 'activity_logs' | 'property_data' | 'complete') => {
    try {
      const newExport = await DataRetentionAPI.requestDataExport(exportType);
      
      setExports(prev => [newExport, ...prev]);
      
      // If export is completed immediately, trigger download
      if (newExport.status === 'completed') {
        // Auto-download the file immediately
        await downloadExport(newExport.id, `data_export_${exportType}_${newExport.id}.zip`);
      }
      
      return newExport;
    } catch (err) {
      console.error('Export request error:', err);
      throw new Error(err instanceof Error ? err.message : 'Failed to request data export');
    }
  };

  const downloadExport = async (id: number, filename?: string) => {
    try {
      const blob = await DataRetentionAPI.downloadDataExport(id);
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = filename || `data_export_${id}.zip`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);
    } catch (err) {
      throw new Error(err instanceof Error ? err.message : 'Failed to download export');
    }
  };

  useEffect(() => {
    fetchExports();
  }, [fetchExports]);

  return {
    exports,
    loading,
    error,
    refetch: fetchExports,
    requestExport,
    downloadExport
  };
};

// Hook for user data summary
export const useUserDataSummary = () => {
  const [summary, setSummary] = useState<UserDataSummary | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchSummary = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      const data = await DataRetentionAPI.getUserDataSummary();
      setSummary(data);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch user data summary');
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchSummary();
  }, [fetchSummary]);

  return {
    summary,
    loading,
    error,
    refetch: fetchSummary
  };
};

// Hook for retention statistics
export const useRetentionStats = () => {
  const [stats, setStats] = useState<RetentionStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchStats = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      const data = await DataRetentionAPI.getRetentionStats();
      setStats(data);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch retention stats');
    } finally {
      setLoading(false);
    }
  }, []);

  const runCleanup = async () => {
    try {
      const result = await DataRetentionAPI.runDataCleanup();
      await fetchStats(); // Refresh stats after cleanup
      return result;
    } catch (err) {
      throw new Error(err instanceof Error ? err.message : 'Failed to run data cleanup');
    }
  };

  useEffect(() => {
    fetchStats();
  }, [fetchStats]);

  return {
    stats,
    loading,
    error,
    refetch: fetchStats,
    runCleanup
  };
};

// Hook for consent management
export const useConsent = () => {
  const [consents, setConsents] = useState<any[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const fetchConsents = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      const data = await DataRetentionAPI.getUserConsents();
      setConsents(data.consents);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch consents');
    } finally {
      setLoading(false);
    }
  }, []);

  const recordConsent = async (consent: {
    consent_type: 'data_processing' | 'marketing' | 'analytics' | 'third_party_sharing';
    granted: boolean;
  }) => {
    try {
      const result = await DataRetentionAPI.recordConsent(consent);
      await fetchConsents(); // Refresh consents
      return result;
    } catch (err) {
      throw new Error(err instanceof Error ? err.message : 'Failed to record consent');
    }
  };

  useEffect(() => {
    fetchConsents();
  }, [fetchConsents]);

  return {
    consents,
    loading,
    error,
    refetch: fetchConsents,
    recordConsent
  };
};