import { useState } from 'react';
import axios from 'axios';
import {
  EntitlementCheckRequest,
  EntitlementCheckResponse,
  FeatureUsageRequest,
  FeatureUsageResponse,
  UserEntitlement,
  Feature,
  PackageEntitlement,
  EntitlementAnalytics
} from '../types/entitlement';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8000';

export const useEntitlements = () => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const checkEntitlement = async (request: EntitlementCheckRequest): Promise<EntitlementCheckResponse | null> => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.post(`${API_URL}/api/entitlements/check`, request);
      return response.data;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to check entitlement');
      return null;
    } finally {
      setLoading(false);
    }
  };

  const checkBulkEntitlements = async (
    userId: number,
    userType: string,
    featureCodes: string[]
  ): Promise<Record<string, EntitlementCheckResponse> | null> => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.post(`${API_URL}/api/entitlements/check-bulk`, {
        user_id: userId,
        user_type: userType,
        feature_codes: featureCodes
      });
      return response.data.entitlements;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to check entitlements');
      return null;
    } finally {
      setLoading(false);
    }
  };

  const useFeature = async (request: FeatureUsageRequest): Promise<FeatureUsageResponse | null> => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.post(`${API_URL}/api/entitlements/use`, request);
      return response.data;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to use feature');
      return null;
    } finally {
      setLoading(false);
    }
  };

  const getMyEntitlements = async (userId: number, userType: string): Promise<UserEntitlement[]> => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.get(`${API_URL}/api/entitlements/my-entitlements`, {
        params: { user_id: userId, user_type: userType }
      });
      return response.data.data;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to fetch entitlements');
      return [];
    } finally {
      setLoading(false);
    }
  };

  // Admin functions
  const getAllFeatures = async (category?: string): Promise<Feature[]> => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.get(`${API_URL}/api/entitlements/admin/features`, {
        params: category ? { category } : {}
      });
      return response.data.data;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to fetch features');
      return [];
    } finally {
      setLoading(false);
    }
  };

  const createFeature = async (feature: Partial<Feature>): Promise<boolean> => {
    setLoading(true);
    setError(null);
    try {
      await axios.post(`${API_URL}/api/entitlements/admin/features`, feature);
      return true;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to create feature');
      return false;
    } finally {
      setLoading(false);
    }
  };

  const toggleFeature = async (featureCode: string): Promise<boolean> => {
    setLoading(true);
    setError(null);
    try {
      await axios.patch(`${API_URL}/api/entitlements/admin/features/${featureCode}/toggle`);
      return true;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to toggle feature');
      return false;
    } finally {
      setLoading(false);
    }
  };

  const getPackageEntitlements = async (packageId?: number): Promise<PackageEntitlement[]> => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.get(`${API_URL}/api/entitlements/admin/package-entitlements`, {
        params: packageId ? { package_id: packageId } : {}
      });
      return response.data.data;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to fetch package entitlements');
      return [];
    } finally {
      setLoading(false);
    }
  };

  const addPackageEntitlement = async (entitlement: Partial<PackageEntitlement>): Promise<boolean> => {
    setLoading(true);
    setError(null);
    try {
      await axios.post(`${API_URL}/api/entitlements/admin/package-entitlements`, entitlement);
      return true;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to add entitlement');
      return false;
    } finally {
      setLoading(false);
    }
  };

  const removePackageEntitlement = async (entitlementId: number): Promise<boolean> => {
    setLoading(true);
    setError(null);
    try {
      await axios.delete(`${API_URL}/api/entitlements/admin/package-entitlements/${entitlementId}`);
      return true;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to remove entitlement');
      return false;
    } finally {
      setLoading(false);
    }
  };

  const getAnalytics = async (featureCode?: string, days: number = 30): Promise<EntitlementAnalytics[]> => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.get(`${API_URL}/api/entitlements/admin/analytics`, {
        params: { feature_code: featureCode, days }
      });
      return response.data.data;
    } catch (err: any) {
      setError(err.response?.data?.detail || 'Failed to fetch analytics');
      return [];
    } finally {
      setLoading(false);
    }
  };

  return {
    loading,
    error,
    checkEntitlement,
    checkBulkEntitlements,
    useFeature,
    getMyEntitlements,
    getAllFeatures,
    createFeature,
    toggleFeature,
    getPackageEntitlements,
    addPackageEntitlement,
    removePackageEntitlement,
    getAnalytics
  };
};
