
import { useState, useCallback } from 'react';
import { api } from '../utils/api';

interface GeocodingResult {
  latitude: number;
  longitude: number;
  displayName: string;
  address: any;
}

interface GeocodingError {
  error: string;
}

export const useGeocoding = () => {
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const geocodeAddress = useCallback(async (params: {
    landmark?: string;
    pincode?: string;
    city?: string;
    state?: string;
  }): Promise<GeocodingResult | null> => {
    // Don't geocode if no meaningful input is provided
    if (!params.landmark && !params.pincode && !params.city) {
      return null;
    }

    // Clear previous errors
    setError(null);
    setIsLoading(true);

    try {
      const result = await api.geocodeAddress(params);
      
      if (result.success && result.latitude && result.longitude) {
        return {
          latitude: result.latitude,
          longitude: result.longitude,
          displayName: result.displayName || 'Location found',
          address: result.address || {}
        };
      } else {
        // Only set error for meaningful failures, not empty inputs
        const hasInput = (params.landmark && params.landmark.length > 2) || 
                        (params.pincode && params.pincode.length >= 6);
        if (hasInput) {
          setError(result.error || 'Location not found. Please check your landmark or pincode.');
        }
        return null;
      }
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Network error occurred';
      setError(errorMessage);
      return null;
    } finally {
      setIsLoading(false);
    }
  }, []);

  const reverseGeocode = useCallback(async (latitude: number, longitude: number) => {
    setIsLoading(true);
    setError(null);

    try {
      const result = await api.reverseGeocode(latitude, longitude);
      
      if (result.success) {
        return {
          displayName: result.displayName,
          address: result.address
        };
      } else {
        setError(result.error || 'Reverse geocoding failed');
        return null;
      }
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Unknown error occurred';
      setError(errorMessage);
      return null;
    } finally {
      setIsLoading(false);
    }
  }, []);

  const clearError = useCallback(() => {
    setError(null);
  }, []);

  return {
    geocodeAddress,
    reverseGeocode,
    isLoading,
    error,
    clearError
  };
};