import { useState, useEffect ,useRef} from 'react';

export interface LegalHealthData {
  legal_status: 'NOT_STARTED' | 'INCOMPLETE' | 'COMPLETE';
  risk_level: 'LOW' | 'MEDIUM' | 'HIGH';
  completion_percentage: number;
  required_documents: number;
  uploaded_documents: number;
  missing_documents: string[];
  document_checklist: Array<{
    type: string;
    display_name: string;
    uploaded: boolean;
    required: boolean;
    description: string;
  }>;
  property_type?: string;
}

export const useLegalHealth = (propertyId: string | number) => {
  const [legalHealth, setLegalHealth] = useState<LegalHealthData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const isMountedRef = useRef(true);

  const fetchLegalHealth = async () => {
  if (!propertyId || !isMountedRef.current) return;

  try {
    setLoading(true);

    const API_BASE =
      import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

    const response = await fetch(
      `${API_BASE}/legal-health/property/${propertyId}/legal-health`,
      {
        headers: { 'Content-Type': 'application/json' }
      }
    );

    if (!isMountedRef.current) return;

    if (
      response.status === 401 ||
      response.status === 403 ||
      response.status === 404
    ) {
      setLegalHealth({
        legal_status: 'NOT_STARTED',
        risk_level: 'HIGH',
        completion_percentage: 0,
        required_documents: 8,
        uploaded_documents: 0,
        missing_documents: [],
        document_checklist: []
      });
      setError(null);
      return;
    }

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }

    const data = await response.json();
    if (isMountedRef.current) {
      setLegalHealth(data);
      setError(null);
    }
  } catch (err) {
    if (!isMountedRef.current) return;

    setLegalHealth({
      legal_status: 'NOT_STARTED',
      risk_level: 'HIGH',
      completion_percentage: 0,
      required_documents: 8,
      uploaded_documents: 0,
      missing_documents: [],
      document_checklist: []
    });
    setError(null);
  } finally {
    if (isMountedRef.current) {
      setLoading(false);
    }
  }
};

  useEffect(() => {
  isMountedRef.current = true;
  fetchLegalHealth();

  return () => {
    isMountedRef.current = false;
  };
}, [propertyId]);


  return {
    legalHealth,
    loading,
    error,
    refetch: fetchLegalHealth
  };
};