import { useState, useCallback } from 'react';
import { BANKS_DATA, getBankById, getBankRate } from '../data/banks';

export interface LoanCalculation {
  emi: number;
  totalInterest: number;
  totalAmount: number;
  monthlyRate: number;
  numPayments: number;
}

export interface BankComparison {
  bankId: string;
  bankName: string;
  interestRate: number;
  processingFee: string;
  emi: number;
  totalInterest: number;
  totalAmount: number;
}

export const useLoanCalculator = () => {
  const [loading, setLoading] = useState(false);
  
  const calculateEMI = useCallback((
    principal: number,
    annualRate: number,
    tenureYears: number
  ): LoanCalculation => {
    const monthlyRate = annualRate / 12 / 100;
    const numPayments = tenureYears * 12;
    
    const emi = (principal * monthlyRate * Math.pow(1 + monthlyRate, numPayments)) / 
                (Math.pow(1 + monthlyRate, numPayments) - 1);
    
    const totalAmount = emi * numPayments;
    const totalInterest = totalAmount - principal;
    
    return {
      emi: Math.round(emi),
      totalInterest: Math.round(totalInterest),
      totalAmount: Math.round(totalAmount),
      monthlyRate,
      numPayments
    };
  }, []);
  
  const compareAllBanks = useCallback((
    loanAmount: number,
    tenure: number
  ): BankComparison[] => {
    setLoading(true);
    
    try {
      const comparisons = BANKS_DATA.map(bank => {
        const rate = getBankRate(bank.id, tenure);
        const calculation = calculateEMI(loanAmount, rate, tenure);
        
        return {
          bankId: bank.id,
          bankName: bank.name,
          interestRate: rate,
          processingFee: bank.processingFee,
          ...calculation
        };
      });
      
      // Sort by EMI (lowest first)
      return comparisons.sort((a, b) => a.emi - b.emi);
    } finally {
      setLoading(false);
    }
  }, [calculateEMI]);
  
  const getBankDetails = useCallback((bankId: string) => {
    return getBankById(bankId);
  }, []);
  
  const getBankRateForTenure = useCallback((bankId: string, tenure: number) => {
    return getBankRate(bankId, tenure);
  }, []);
  
  return {
    calculateEMI,
    compareAllBanks,
    getBankDetails,
    getBankRateForTenure,
    loading,
    banks: BANKS_DATA
  };
};