import { useState, useCallback, useEffect } from 'react';
import { negotiationApi } from '../api/negotiation.api';
import { NegotiationState, NegotiationTimeline, UserRole } from '../types/negotiation';

export const useNegotiation = (negotiationId: string, userRole: UserRole) => {
  const [state, setState] = useState<NegotiationState>({
    timeline: null,
    loading: false,
    error: null,
  });

  // Fetch timeline data
  const fetchTimeline = useCallback(async () => {
    if (!negotiationId) return;

    setState(prev => ({ ...prev, loading: true, error: null }));
    
    try {
      const timeline = await negotiationApi.getTimeline(negotiationId);
      setState(prev => ({ ...prev, timeline, loading: false }));
    } catch (error: any) {
      const errorMessage = error.response?.data?.detail || error.message || 'Failed to fetch timeline';
      setState(prev => ({ ...prev, error: errorMessage, loading: false }));
    }
  }, [negotiationId]);

  // Submit offer
  const submitOffer = useCallback(async (amount: number, message?: string) => {
    if (!negotiationId) throw new Error('No negotiation ID');

    setState(prev => ({ ...prev, loading: true, error: null }));

    try {
      await negotiationApi.submitOffer(negotiationId, { amount, message });
      // Refresh timeline after successful submission
      await fetchTimeline();
    } catch (error: any) {
      const errorMessage = error.response?.data?.detail || error.message || 'Failed to submit offer';
      setState(prev => ({ ...prev, error: errorMessage, loading: false }));
      throw error;
    }
  }, [negotiationId, fetchTimeline]);

  // Accept offer
  const acceptOffer = useCallback(async (eventId: string) => {
    setState(prev => ({ ...prev, loading: true, error: null }));

    try {
      await negotiationApi.acceptOffer(eventId);
      // Refresh timeline after successful acceptance
      await fetchTimeline();
    } catch (error: any) {
      const errorMessage = error.response?.data?.detail || error.message || 'Failed to accept offer';
      setState(prev => ({ ...prev, error: errorMessage, loading: false }));
      throw error;
    }
  }, [fetchTimeline]);

  // Reject offer
  const rejectOffer = useCallback(async (eventId: string, reason?: string) => {
    setState(prev => ({ ...prev, loading: true, error: null }));

    try {
      await negotiationApi.rejectOffer(eventId, reason ? { reason } : undefined);
      // Refresh timeline after successful rejection
      await fetchTimeline();
    } catch (error: any) {
      const errorMessage = error.response?.data?.detail || error.message || 'Failed to reject offer';
      setState(prev => ({ ...prev, error: errorMessage, loading: false }));
      throw error;
    }
  }, [fetchTimeline]);

  // Refresh timeline
  const refresh = useCallback(() => {
    fetchTimeline();
  }, [fetchTimeline]);

  // Auto-fetch on mount
  useEffect(() => {
    fetchTimeline();
  }, [fetchTimeline]);

  // Helper to determine if user can perform actions
  const canAcceptOffer = useCallback((event: any) => {
    if (!event.is_active || !event.amount) return false;
    
    // User can accept if they're not the one who made the offer
    if (userRole === 'BUYER' && event.offered_by !== 'BUYER') return true;
    if ((userRole === 'SELLER' || userRole === 'AGENT') && event.offered_by === 'BUYER') return true;
    
    return false;
  }, [userRole]);

  const canRejectOffer = useCallback((event: any) => {
    return canAcceptOffer(event); // Same logic as accept
  }, [canAcceptOffer]);

  const canMakeOffer = useCallback(() => {
    if (!state.timeline) return false;
    
    const currentStatus = state.timeline.status;
    return !['ACCEPTED', 'REJECTED', 'EXPIRED'].includes(currentStatus);
  }, [state.timeline]);

  return {
    ...state,
    submitOffer,
    acceptOffer,
    rejectOffer,
    refresh,
    canAcceptOffer,
    canRejectOffer,
    canMakeOffer,
  };
};