import { useState, useEffect } from 'react';
import { Coupon, DiscountRule, ScholarshipProgram, ScholarshipApplication, PriceCalculation } from '../types/promotions';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090';

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    'Content-Type': 'application/json',
    'Authorization': token ? `Bearer ${token}` : ''
  };
};

// ==================== ADMIN HOOKS ====================

export const useCoupons = () => {
  const [coupons, setCoupons] = useState<Coupon[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchCoupons = async () => {
    try {
      setLoading(true);
      const response = await fetch(`${API_URL}/api/admin/coupons/`, {
        headers: getAuthHeaders()
      });
      const data = await response.json();
      if (data.success) setCoupons(data.data);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchCoupons(); }, []);

  return { coupons, loading, error, refetch: fetchCoupons };
};

export const useDiscountRules = () => {
  const [rules, setRules] = useState<DiscountRule[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchRules = async () => {
    try {
      setLoading(true);
      const response = await fetch(`${API_URL}/api/admin/discount-rules/`, {
        headers: getAuthHeaders()
      });
      const data = await response.json();
      if (data.success) setRules(data.data);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchRules(); }, []);

  return { rules, loading, error, refetch: fetchRules };
};

export const useScholarshipPrograms = () => {
  const [programs, setPrograms] = useState<ScholarshipProgram[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchPrograms = async () => {
    try {
      setLoading(true);
      const response = await fetch(`${API_URL}/api/admin/scholarships/`, {
        headers: getAuthHeaders()
      });
      const data = await response.json();
      if (data.success) setPrograms(data.data);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchPrograms(); }, []);

  return { programs, loading, error, refetch: fetchPrograms };
};

export const useScholarshipApplications = (status?: string) => {
  const [applications, setApplications] = useState<ScholarshipApplication[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchApplications = async () => {
    try {
      setLoading(true);
      const url = status 
        ? `${API_URL}/api/admin/scholarship-applications/?status=${status}`
        : `${API_URL}/api/admin/scholarship-applications/`;
      const response = await fetch(url, {
        headers: getAuthHeaders()
      });
      const data = await response.json();
      if (data.success) setApplications(data.data);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchApplications(); }, [status]);

  return { applications, loading, error, refetch: fetchApplications };
};

// ==================== USER HOOKS ====================

export const useActiveScholarships = () => {
  const [programs, setPrograms] = useState<ScholarshipProgram[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchPrograms = async () => {
    try {
      setLoading(true);
      const response = await fetch(`${API_URL}/api/user/scholarships/`, {
        headers: getAuthHeaders()
      });
      const data = await response.json();
      if (data.success) setPrograms(data.data);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchPrograms(); }, []);

  return { programs, loading, error, refetch: fetchPrograms };
};

export const useMyScholarshipApplications = (userId: number) => {
  const [applications, setApplications] = useState<ScholarshipApplication[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchApplications = async () => {
    try {
      setLoading(true);
      const response = await fetch(`${API_URL}/api/user/scholarships/my-applications/?user_id=${userId}`, {
        headers: getAuthHeaders()
      });
      const data = await response.json();
      if (data.success) setApplications(data.data);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { if (userId) fetchApplications(); }, [userId]);

  return { applications, loading, error, refetch: fetchApplications };
};

export const useMyActiveScholarship = (userId: number) => {
  const [scholarship, setScholarship] = useState<ScholarshipApplication | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchScholarship = async () => {
    try {
      setLoading(true);
      const response = await fetch(`${API_URL}/api/user/scholarships/my-active/?user_id=${userId}`, {
        headers: getAuthHeaders()
      });
      const data = await response.json();
      if (data.success) setScholarship(data.data);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { if (userId) fetchScholarship(); }, [userId]);

  return { scholarship, loading, error, refetch: fetchScholarship };
};

// ==================== UTILITY FUNCTIONS ====================

export const validateCoupon = async (couponCode: string, planId: number, userId: number): Promise<any> => {
  const response = await fetch(`${API_URL}/api/user/validate-coupon/?user_id=${userId}`, {
    method: 'POST',
    headers: getAuthHeaders(),
    body: JSON.stringify({ coupon_code: couponCode, plan_id: planId, original_price: 0 })
  });
  return response.json();
};

export const calculatePrice = async (
  planId: number, 
  userId: number, 
  originalPrice: number, 
  couponCode?: string,
  context?: Record<string, any>
): Promise<PriceCalculation> => {
  const response = await fetch(`${API_URL}/api/user/calculate-price/`, {
    method: 'POST',
    headers: getAuthHeaders(),
    body: JSON.stringify({ plan_id: planId, user_id: userId, original_price: originalPrice, coupon_code: couponCode, context })
  });
  const data = await response.json();
  return data.data;
};
