import { useState, useEffect } from 'react';

const COMPARISON_STORAGE_KEY = 'propertyComparison';
const COMPARISON_CHANGE_EVENT = 'comparisonChange';

export interface ComparisonProperty {
  id: number;
  title: string;
  locality: string;
  city: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  sale_price?: string;
  photo?: string;
  furnishing?: string;
  possession_status?: string;
  amenities?: string[];
}

export const usePropertyComparison = () => {
  const [comparisonList, setComparisonList] = useState<ComparisonProperty[]>([]);

  useEffect(() => {
    loadComparison();
    
    const handleComparisonChange = () => {
      loadComparison();
    };

    window.addEventListener(COMPARISON_CHANGE_EVENT, handleComparisonChange);
    return () => window.removeEventListener(COMPARISON_CHANGE_EVENT, handleComparisonChange);
  }, []);

  const loadComparison = () => {
    try {
      const stored = localStorage.getItem(COMPARISON_STORAGE_KEY);
      if (stored) {
        setComparisonList(JSON.parse(stored));
      }
    } catch (error) {
      console.error('Error loading comparison:', error);
    }
  };

  const saveComparison = (properties: ComparisonProperty[]) => {
    try {
      localStorage.setItem(COMPARISON_STORAGE_KEY, JSON.stringify(properties));
      setComparisonList(properties);
      window.dispatchEvent(new Event(COMPARISON_CHANGE_EVENT));
    } catch (error) {
      console.error('Error saving comparison:', error);
    }
  };

  const addToComparison = (property: ComparisonProperty) => {
    if (comparisonList.length >= 3) {
      throw new Error('Maximum 3 properties can be compared');
    }
    
    if (comparisonList.some(p => p.id === property.id)) {
      throw new Error('Property already in comparison');
    }

    const updated = [...comparisonList, property];
    saveComparison(updated);
    return true;
  };

  const removeFromComparison = (propertyId: number) => {
    const updated = comparisonList.filter(p => p.id !== propertyId);
    saveComparison(updated);
  };

  const clearComparison = () => {
    saveComparison([]);
  };

  const isInComparison = (propertyId: number) => {
    return comparisonList.some(p => p.id === propertyId);
  };

  return {
    comparisonList,
    addToComparison,
    removeFromComparison,
    clearComparison,
    isInComparison,
    canAddMore: comparisonList.length < 3
  };
};