import { useState, useEffect } from 'react';
import { wishlistAPI } from '../api/user/wishlistApi';
import { userDashboardAPI } from '../api/user/userDashboard';
import { getUserId } from '../utils/userUtils';

// Custom event for cross-component communication
const PROPERTY_STATE_CHANGE = 'propertyStateChange';

interface PropertyStateEvent extends CustomEvent {
  detail: {
    propertyId: number;
    type: 'wishlist' | 'saved';
    isActive: boolean;
  };
}

// Dispatch property state change event
export const dispatchPropertyStateChange = (propertyId: number, type: 'wishlist' | 'saved', isActive: boolean) => {
  window.dispatchEvent(new CustomEvent(PROPERTY_STATE_CHANGE, {
    detail: { propertyId, type, isActive }
  }));
};

// Hook for wishlist properties (API)
export const useWishlistProperty = (propertyId: number) => {
  const [isWishlisted, setIsWishlisted] = useState(false);

  const checkWishlistStatus = async () => {
    const userId = getUserId();
    if (!userId) return false;
    
    try {
      const wishlist = await wishlistAPI.getWishlist(userId);
      const wishlisted = wishlist.some(item => item.property_id === propertyId);
      setIsWishlisted(wishlisted);
      return wishlisted;
    } catch (error) {
      console.error('Error checking wishlist status:', error);
      return false;
    }
  };

  const toggleWishlist = async () => {
    const userId = getUserId();
    if (!userId) {
      throw new Error('Please login to add to wishlist');
    }
    
    try {
      let newWishlistState: boolean;
      
      if (isWishlisted) {
        await wishlistAPI.removeFromWishlist(userId, propertyId);
        newWishlistState = false;
      } else {
        await wishlistAPI.addToWishlist(userId, propertyId);
        newWishlistState = true;
      }
      
      setIsWishlisted(newWishlistState);
      dispatchPropertyStateChange(propertyId, 'wishlist', newWishlistState);
      return newWishlistState;
    } catch (error) {
      console.error('Error toggling wishlist:', error);
      throw error;
    }
  };

  useEffect(() => {
    checkWishlistStatus();

    const handleStateChange = (event: Event) => {
      const customEvent = event as PropertyStateEvent;
      if (customEvent.detail.propertyId === propertyId && customEvent.detail.type === 'wishlist') {
        setIsWishlisted(customEvent.detail.isActive);
      }
    };

    window.addEventListener(PROPERTY_STATE_CHANGE, handleStateChange);
    return () => window.removeEventListener(PROPERTY_STATE_CHANGE, handleStateChange);
  }, [propertyId]);

  return { isWishlisted, toggleWishlist, checkWishlistStatus };
};

// Hook for saved properties (API)
export const useSavedProperty = (propertyId: number) => {
  const [isSaved, setIsSaved] = useState(false);

  const checkSavedStatus = async () => {
    const userId = getUserId();
    if (!userId) return false;
    
    try {
      const savedProperties = await userDashboardAPI.getSavedProperties(userId);
      const saved = savedProperties.some(item => item.property_id === propertyId);
      setIsSaved(saved);
      return saved;
    } catch (error) {
      if (error instanceof Error && error.message.includes('Failed to fetch')) {
        console.warn('Backend not available - bookmark status unavailable');
      } else {
        console.error('Error checking saved status:', error);
      }
      return false;
    }
  };

  const toggleSaved = async () => {
    const userId = getUserId();
    if (!userId) {
      throw new Error('Please login to save properties');
    }
    
    try {
      let newSavedState: boolean;
      
      if (isSaved) {
        await userDashboardAPI.removeSavedProperty(userId, propertyId);
        newSavedState = false;
      } else {
        await userDashboardAPI.saveProperty(userId, propertyId);
        newSavedState = true;
      }
      
      setIsSaved(newSavedState);
      dispatchPropertyStateChange(propertyId, 'saved', newSavedState);
      return newSavedState;
    } catch (error) {
      console.error('Error toggling saved status:', error);
      throw error;
    }
  };

  useEffect(() => {
    checkSavedStatus();

    const handleStateChange = (event: Event) => {
      const customEvent = event as PropertyStateEvent;
      if (customEvent.detail.propertyId === propertyId && customEvent.detail.type === 'saved') {
        setIsSaved(customEvent.detail.isActive);
      }
    };

    window.addEventListener(PROPERTY_STATE_CHANGE, handleStateChange);
    return () => window.removeEventListener(PROPERTY_STATE_CHANGE, handleStateChange);
  }, [propertyId]);

  return { isSaved, toggleSaved, checkSavedStatus };
};