/**
 * RUTHLESS SESSION GUARD HOOK - ZERO TOLERANCE FOR EXPIRED DATA
 * Protects components from displaying stale/cached data
 */

import { useEffect, useState, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import { ruthlessSessionValidator, SessionStatus } from '../utils/sessionValidator';

interface SessionGuardState {
  isValidating: boolean;
  sessionValid: boolean;
  sessionExpired: boolean;
  remainingTime: number;
  error: string | null;
}

export const useRuthlessSessionGuard = (requireSession: boolean = true) => {
  const navigate = useNavigate();
  const [state, setState] = useState<SessionGuardState>({
    isValidating: true,
    sessionValid: false,
    sessionExpired: false,
    remainingTime: 0,
    error: null
  });

  /**
   * RUTHLESS SESSION VALIDATION - NO MERCY
   */
  const validateSession = useCallback(async () => {
    if (!requireSession) {
      setState(prev => ({ ...prev, isValidating: false, sessionValid: true }));
      return;
    }

    try {
      setState(prev => ({ ...prev, isValidating: true, error: null }));

      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id || user.id;
      const sessionId = new URLSearchParams(window.location.search).get('session') || 
                       localStorage.getItem('verification-session-id');

      if (!userId || !sessionId) {
        // NO SESSION DATA - IMMEDIATE TERMINATION
        setState({
          isValidating: false,
          sessionValid: false,
          sessionExpired: true,
          remainingTime: 0,
          error: 'No valid session found'
        });
        
        if (requireSession) {
          setTimeout(() => navigate('/'), 1000);
        }
        return;
      }

      // RUTHLESS VALIDATION
      const sessionStatus: SessionStatus = await ruthlessSessionValidator.validateSession(sessionId, userId);

      if (sessionStatus.expired || !sessionStatus.valid) {
        // SESSION EXPIRED - NUCLEAR OPTION
        setState({
          isValidating: false,
          sessionValid: false,
          sessionExpired: true,
          remainingTime: 0,
          error: 'Session expired - all data destroyed'
        });
        
        // Force redirect after brief delay
        setTimeout(() => navigate('/'), 2000);
        return;
      }

      // SESSION VALID - PROCEED WITH CAUTION
      setState({
        isValidating: false,
        sessionValid: true,
        sessionExpired: false,
        remainingTime: sessionStatus.remainingTime,
        error: null
      });

      // Set up periodic validation for active sessions
      if (sessionStatus.remainingTime > 0) {
        setTimeout(validateSession, Math.min(5 * 60 * 1000, sessionStatus.remainingTime / 2));
      }

    } catch (error) {
      console.error('RUTHLESS SESSION GUARD FAILED:', error);
      setState({
        isValidating: false,
        sessionValid: false,
        sessionExpired: true,
        remainingTime: 0,
        error: 'Session validation failed'
      });
      
      if (requireSession) {
        setTimeout(() => navigate('/'), 1000);
      }
    }
  }, [requireSession, navigate]);

  /**
   * FORCE SESSION TERMINATION
   */
  const terminateSession = useCallback(() => {
    localStorage.removeItem('verification-session-id');
    localStorage.removeItem('owner-details');
    sessionStorage.clear();
    navigate('/');
  }, [navigate]);

  /**
   * GET FORMATTED TIME REMAINING
   */
  const getFormattedTimeRemaining = useCallback(() => {
    if (state.remainingTime <= 0) return 'EXPIRED';
    return ruthlessSessionValidator.formatRemainingTime(state.remainingTime);
  }, [state.remainingTime]);

  // Initialize validation on mount
  useEffect(() => {
    validateSession();
  }, [validateSession]);

  return {
    ...state,
    validateSession,
    terminateSession,
    getFormattedTimeRemaining,
    isNearExpiration: ruthlessSessionValidator.isNearExpiration(state.remainingTime)
  };
};