import { useState, useCallback } from 'react';
import { saveProperty, removeSavedProperty } from '../api/user/userDashboard';

export const useSavedProperties = () => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleSaveProperty = useCallback(async (propertyId: number) => {
    setLoading(true);
    setError(null);
    
    try {
      await saveProperty(propertyId);
      return { success: true, message: 'Property saved successfully!' };
    } catch (err: any) {
      const errorMessage = err.message || 'Failed to save property';
      setError(errorMessage);
      return { success: false, message: errorMessage };
    } finally {
      setLoading(false);
    }
  }, []);

  const handleRemoveProperty = useCallback(async (propertyId: number) => {
    setLoading(true);
    setError(null);
    
    try {
      await removeSavedProperty(propertyId);
      return { success: true, message: 'Property removed from saved list!' };
    } catch (err: any) {
      const errorMessage = err.message || 'Failed to remove property';
      setError(errorMessage);
      return { success: false, message: errorMessage };
    } finally {
      setLoading(false);
    }
  }, []);

  return {
    loading,
    error,
    saveProperty: handleSaveProperty,
    removeProperty: handleRemoveProperty,
  };
};