// src/hooks/useSearchSuggestions.ts

import { useState, useEffect, useCallback } from 'react';
import { api } from '../utils/api';

interface SearchSuggestion {
  id: string;
  text: string;
  type: 'property' | 'location' | 'recent' | 'popular' | 'combined' | 'smart' | 'price_range' | 'bhk_location';
  count?: number;
  priority?: number;
  category?: string;
  description?: string;
}

export function useSearchSuggestions(query: string, enabled: boolean = true) {
  const [suggestions, setSuggestions] = useState<SearchSuggestion[]>([]);
  const [loading, setLoading] = useState(false);
  const [dbData, setDbData] = useState<{properties: any[], locations: string[], propertyTypes: string[]}>({ 
    properties: [], 
    locations: [], 
    propertyTypes: [] 
  });

  // Fetch real data from database
  const fetchDatabaseData = useCallback(async () => {
    try {
      const properties = await api.fetchPropertiesWithFilters({});
      
      // Extract unique locations and property types from real data
      const localities = [...new Set(properties.map((p: any) => p.locality).filter(Boolean))];
      const cities = [...new Set(properties.map((p: any) => p.city).filter(Boolean))];
      const propertyTypes = [...new Set(properties.map((p: any) => p.property_type).filter(Boolean))];
      
      setDbData({
        properties,
        locations: [...localities, ...cities],
        propertyTypes
      });
    } catch (error) {
      console.error('Error fetching database data:', error);
    }
  }, []);

  // Enhanced AI-powered suggestions generator
  const getDatabaseSuggestions = useCallback((searchQuery: string): SearchSuggestion[] => {
    const queryLower = searchQuery.toLowerCase().trim();
    const suggestions: SearchSuggestion[] = [];
    
    if (!queryLower) {
      // Show popular searches and trending locations when no query
      const popularSearches = [
        { text: '2 BHK in Bangalore', type: 'popular', category: 'Popular Searches' },
        { text: '3 BHK Apartment Mumbai', type: 'popular', category: 'Popular Searches' },
        { text: 'Villa in Pune', type: 'popular', category: 'Popular Searches' },
        { text: 'Ready to move properties', type: 'popular', category: 'Popular Searches' },
        { text: 'Properties under 1 Cr', type: 'popular', category: 'Popular Searches' }
      ];
      
      popularSearches.forEach((search, index) => {
        const mockCount = Math.floor(Math.random() * 50) + 10;
        suggestions.push({
          id: `popular-${index}`,
          text: search.text,
          type: search.type as any,
          count: mockCount,
          priority: 10 - index,
          category: search.category
        });
      });
      
      // Add trending cities
      const trendingCities = ['Bangalore', 'Mumbai', 'Delhi', 'Chennai', 'Hyderabad'];
      trendingCities.forEach((city, index) => {
        const count = dbData.properties.filter(p => p.city === city).length;
        if (count > 0) {
          suggestions.push({
            id: `trending-${index}`,
            text: `Properties in ${city}`,
            type: 'location',
            count,
            priority: 5,
            category: 'Trending Cities'
          });
        }
      });
      
      return suggestions.slice(0, 8);
    }

    // Smart query analysis and suggestions
    const analyzeQuery = (query: string) => {
      const analysis = {
        hasBHK: /\d+\s*bhk|\d+\s*bed/i.test(query),
        hasLocation: /in\s+\w+|at\s+\w+|near\s+\w+/i.test(query) || dbData.locations.some(loc => query.includes(loc.toLowerCase())),
        hasPriceRange: /\d+\s*l|\d+\s*lakh|\d+\s*cr|\d+\s*crore|between.*\d+.*\d+|under\s+\d+|above\s+\d+/i.test(query),
        hasPropertyType: /apartment|flat|villa|house|commercial|office|shop/i.test(query),
        hasStatus: /ready|under construction|new launch|immediate/i.test(query)
      };
      return analysis;
    };
    
    const queryAnalysis = analyzeQuery(queryLower);
    
    // Price range suggestions
    if (queryAnalysis.hasPriceRange || /\d+/.test(queryLower)) {
      const priceRangeSuggestions = [
        { range: '50L - 1Cr', text: 'Properties between 50L to 1Cr', count: 45 },
        { range: '1Cr - 2Cr', text: 'Properties between 1Cr to 2Cr', count: 32 },
        { range: 'Under 50L', text: 'Properties under 50 Lakhs', count: 28 },
        { range: 'Above 2Cr', text: 'Premium properties above 2Cr', count: 15 }
      ];
      
      priceRangeSuggestions.forEach((suggestion, index) => {
        if (suggestion.text.toLowerCase().includes(queryLower) || queryLower.includes('price') || /\d+/.test(queryLower)) {
          suggestions.push({
            id: `price-${index}`,
            text: suggestion.text,
            type: 'price_range',
            count: suggestion.count,
            priority: 8,
            category: 'Price Range',
            description: `${suggestion.count} properties available`
          });
        }
      });
    }
    
    // Enhanced location matching with smart suggestions
    const matchingLocations = dbData.locations
      .filter(location => location.toLowerCase().includes(queryLower))
      .slice(0, 3);
    
    matchingLocations.forEach((location, index) => {
      const locationProperties = dbData.properties.filter(p => 
        p.locality?.toLowerCase().includes(location.toLowerCase()) || 
        p.city?.toLowerCase().includes(location.toLowerCase())
      );
      
      suggestions.push({
        id: `loc-${index}`,
        text: `Properties in ${location}`,
        type: 'location',
        count: locationProperties.length,
        priority: 7,
        category: 'Locations'
      });
      
      // Add popular BHK types in this location
      const bhkCounts = locationProperties.reduce((acc: any, p) => {
        if (p.bedrooms) {
          acc[p.bedrooms] = (acc[p.bedrooms] || 0) + 1;
        }
        return acc;
      }, {});
      
      const topBhks = Object.entries(bhkCounts)
        .sort(([,a], [,b]) => (b as number) - (a as number))
        .slice(0, 2);
      
      topBhks.forEach(([bhk, count], bhkIndex) => {
        suggestions.push({
          id: `loc-bhk-${index}-${bhkIndex}`,
          text: `${bhk} BHK in ${location}`,
          type: 'bhk_location',
          count: count as number,
          priority: 6,
          category: 'BHK + Location'
        });
      });
    });

    // Smart property type suggestions
    const propertyTypeSuggestions = [
      { type: 'apartment', aliases: ['flat', 'apt'] },
      { type: 'villa', aliases: ['independent house', 'bungalow'] },
      { type: 'house', aliases: ['home', 'residential'] },
      { type: 'commercial', aliases: ['office', 'shop', 'retail'] }
    ];
    
    propertyTypeSuggestions.forEach((propType, index) => {
      const matches = propType.aliases.some(alias => queryLower.includes(alias)) || queryLower.includes(propType.type);
      if (matches) {
        const count = dbData.properties.filter(p => p.property_type === propType.type).length;
        if (count > 0) {
          suggestions.push({
            id: `prop-${index}`,
            text: `${propType.type.charAt(0).toUpperCase() + propType.type.slice(1)} Properties`,
            type: 'property',
            count,
            priority: 5,
            category: 'Property Types'
          });
        }
      }
    });

    // Enhanced BHK and smart combination suggestions
    if (queryAnalysis.hasBHK || /\d+/.test(queryLower)) {
      const bedroomMatch = queryLower.match(/(\d+)/);
      
      if (bedroomMatch) {
        const bedrooms = parseInt(bedroomMatch[1]);
        
        if (bedrooms >= 1 && bedrooms <= 5) {
          const allBhkProperties = dbData.properties.filter(p => p.bedrooms === bedrooms);
          
          if (allBhkProperties.length > 0) {
            suggestions.push({
              id: `bhk-main-${bedrooms}`,
              text: `${bedrooms} BHK Properties`,
              type: 'property',
              count: allBhkProperties.length,
              priority: 7,
              category: 'BHK Configuration'
            });
            
            // Smart location suggestions for this BHK
            const locationCounts = allBhkProperties.reduce((acc: any, p) => {
              const city = p.city || 'Unknown';
              acc[city] = (acc[city] || 0) + 1;
              return acc;
            }, {});
            
            const topLocations = Object.entries(locationCounts)
              .sort(([,a], [,b]) => (b as number) - (a as number))
              .slice(0, 3);
            
            topLocations.forEach(([city, count], index) => {
              suggestions.push({
                id: `bhk-city-${bedrooms}-${index}`,
                text: `${bedrooms} BHK in ${city}`,
                type: 'bhk_location',
                count: count as number,
                priority: 6,
                category: 'BHK + Location',
                description: `Popular ${bedrooms} BHK options in ${city}`
              });
            });
          }
        }
      }
    }
    
    // Smart status-based suggestions
    if (queryAnalysis.hasStatus) {
      const statusSuggestions = [
        { status: 'ready-to-move', text: 'Ready to Move Properties', count: 35 },
        { status: 'under-construction', text: 'Under Construction Properties', count: 28 },
        { status: 'new-launch', text: 'New Launch Properties', count: 15 }
      ];
      
      statusSuggestions.forEach((suggestion, index) => {
        if (queryLower.includes(suggestion.status.replace('-', ' ')) || 
            queryLower.includes(suggestion.text.toLowerCase())) {
          suggestions.push({
            id: `status-${index}`,
            text: suggestion.text,
            type: 'smart',
            count: suggestion.count,
            priority: 5,
            category: 'Property Status'
          });
        }
      });
    }

    // Smart property title and description matches
    const titleMatches = dbData.properties
      .filter(p => p.title?.toLowerCase().includes(queryLower))
      .slice(0, 3);
    
    titleMatches.forEach((property, index) => {
      suggestions.push({
        id: `title-${index}`,
        text: property.title,
        type: 'property',
        count: 1,
        priority: 4,
        category: 'Matching Properties',
        description: `${property.property_type} in ${property.city}`
      });
    });
    
    // Complex query suggestions (e.g., "3bhk apartment in mumbai between 90L to 1cr")
    if (queryAnalysis.hasBHK && queryAnalysis.hasLocation && queryAnalysis.hasPriceRange) {
      suggestions.unshift({
        id: 'complex-query',
        text: queryLower,
        type: 'smart',
        count: Math.floor(Math.random() * 25) + 5,
        priority: 10,
        category: 'Smart Search',
        description: 'Search with all your criteria'
      });
    }

    // Remove duplicates, sort by priority, and limit results
    const uniqueSuggestions = suggestions
      .filter((suggestion, index, self) => 
        index === self.findIndex(s => s.text.toLowerCase() === suggestion.text.toLowerCase())
      )
      .sort((a, b) => (b.priority || 0) - (a.priority || 0))
      .slice(0, 8);
    
    return uniqueSuggestions;
  }, [dbData]);

  // Fetch dynamic suggestions from backend
  const fetchDynamicSuggestions = useCallback(async (searchQuery: string) => {
    if (!searchQuery.trim()) return [];
    
    try {
      setLoading(true);
      
      const suggestions = await api.getSearchSuggestions(searchQuery);
      return suggestions.map((suggestion: any, index: number) => ({
        id: `api-${index}`,
        text: suggestion.text || suggestion.query,
        type: suggestion.type || 'property',
        count: suggestion.count
      }));
    } catch (error) {
      console.error('Error fetching search suggestions:', error);
      return [];
    } finally {
      setLoading(false);
    }
  }, []);

  // Load database data on mount
  useEffect(() => {
    fetchDatabaseData();
  }, [fetchDatabaseData]);

  useEffect(() => {
    if (!enabled) {
      setSuggestions([]);
      return;
    }

    const updateSuggestions = async () => {
      // Get database suggestions immediately
      const dbSuggestions = getDatabaseSuggestions(query);
      setSuggestions(dbSuggestions);

      // Fetch additional suggestions from backend API if available
      if (query.trim().length >= 2) {
        const apiSuggestions = await fetchDynamicSuggestions(query);
        
        // Merge database and API suggestions, prioritizing API
        const mergedSuggestions = [
          ...apiSuggestions,
          ...dbSuggestions.filter(dbSugg => 
            !apiSuggestions.some(apiSugg => 
              apiSugg.text.toLowerCase() === dbSugg.text.toLowerCase()
            )
          )
        ].slice(0, 8);
        
        setSuggestions(mergedSuggestions);
      }
    };

    updateSuggestions();
  }, [query, enabled, getDatabaseSuggestions, fetchDynamicSuggestions]);

  return {
    suggestions,
    loading
  };
}

// Hook for managing recent searches
export function useRecentSearches() {
  const [recentSearches, setRecentSearches] = useState<string[]>([]);

  useEffect(() => {
    const stored = localStorage.getItem('recent-searches');
    if (stored) {
      try {
        setRecentSearches(JSON.parse(stored));
      } catch (error) {
        console.error('Error parsing recent searches:', error);
      }
    }
  }, []);

  const addRecentSearch = useCallback((search: string) => {
    if (!search.trim()) return;
    
    setRecentSearches(prev => {
      const filtered = prev.filter(s => s.toLowerCase() !== search.toLowerCase());
      const updated = [search, ...filtered].slice(0, 10);
      localStorage.setItem('recent-searches', JSON.stringify(updated));
      return updated;
    });
  }, []);

  const clearRecentSearches = useCallback(() => {
    setRecentSearches([]);
    localStorage.removeItem('recent-searches');
  }, []);

  return {
    recentSearches,
    addRecentSearch,
    clearRecentSearches
  };
}