import { useState, useEffect } from 'react';
import { sanitizeStateData, validateSession } from '../utils/security';

export const useSecureStorage = <T>(key: string, defaultValue: T) => {
  const [value, setValue] = useState<T>(() => {
    try {
      validateSession();
      const item = localStorage.getItem(key);
      return item ? sanitizeStateData(JSON.parse(item)) : defaultValue;
    } catch {
      return defaultValue;
    }
  });

  const setSecureValue = (newValue: T) => {
    try {
      validateSession();
      const sanitized = sanitizeStateData(newValue);
      setValue(sanitized);
      localStorage.setItem(key, JSON.stringify(sanitized));
    } catch (error) {
      console.error('Secure storage error:', error);
      setValue(defaultValue);
    }
  };

  const removeValue = () => {
    try {
      setValue(defaultValue);
      localStorage.removeItem(key);
    } catch (error) {
      console.error('Remove storage error:', error);
    }
  };

  // Validate session on mount
  useEffect(() => {
    try {
      validateSession();
    } catch {
      removeValue();
    }
  }, []);

  return [value, setSecureValue, removeValue] as const;
};