import { useEffect, useRef } from 'react';
import { useNavigate } from 'react-router-dom';
import { ruthlessSessionValidator } from '../utils/sessionValidator';

interface UseSessionMonitorOptions {
  enabled?: boolean;
  redirectPath?: string;
  checkInterval?: number;
}

export const useSessionMonitor = (options: UseSessionMonitorOptions = {}) => {
  const {
    enabled = true,
    redirectPath = '/',
    checkInterval = 30000 // Check every 30 seconds
  } = options;
  
  const navigate = useNavigate();
  const intervalRef = useRef<NodeJS.Timeout | null>(null);
  const lastWarningRef = useRef<number>(0);

  useEffect(() => {
    if (!enabled) return;

    const checkSession = async () => {
      try {
        const sessionId = localStorage.getItem('verification-session-id');
        const userStr = localStorage.getItem('user');
        
        if (!sessionId) {
          console.warn('No session ID found, redirecting to landing page');
          navigate(redirectPath);
          return;
        }

        let userId = '550e8400-e29b-41d4-a716-446655440000'; // Default fallback
        if (userStr) {
          try {
            const user = JSON.parse(userStr);
            userId = user?.user_id || user?.id || userId;
          } catch (e) {
            console.warn('Error parsing user data');
          }
        }

        const sessionStatus = await ruthlessSessionValidator.validateSession(sessionId, userId);
        
        if (!sessionStatus.valid || sessionStatus.expired) {
          console.warn('Session expired or invalid, redirecting to landing page');
          // Clear all verification data
          localStorage.removeItem('verification-session-id');
          localStorage.removeItem('owner-details');
          localStorage.removeItem('verification-documents');
          localStorage.removeItem('ribil-score');
          localStorage.removeItem('verification-step');
          
          // Show expiration message
          alert('Your verification session has expired. You will be redirected to the home page.');
          navigate(redirectPath);
          return;
        }

        // Show warnings for sessions expiring soon
        const remainingMinutes = Math.floor(sessionStatus.remainingTime / (1000 * 60));
        const now = Date.now();
        
        // Show warning every 5 minutes when less than 30 minutes remain
        if (remainingMinutes <= 30 && remainingMinutes > 0) {
          if (now - lastWarningRef.current > 5 * 60 * 1000) { // 5 minutes since last warning
            lastWarningRef.current = now;
            
            if (remainingMinutes <= 5) {
              // Critical warning
              const shouldContinue = confirm(
                `🚨 CRITICAL: Your session expires in ${remainingMinutes} minute(s)!\n\n` +
                'All verification data will be PERMANENTLY LOST!\n\n' +
                'Click OK to continue or Cancel to save your progress by going to the home page.'
              );
              
              if (!shouldContinue) {
                navigate(redirectPath);
                return;
              }
            } else if (remainingMinutes <= 15) {
              // Final warning
              const shouldContinue = confirm(
                `⚠️ WARNING: Your session expires in ${remainingMinutes} minutes!\n\n` +
                'Please complete your verification soon or your data will be lost.\n\n' +
                'Click OK to continue or Cancel to go to the home page.'
              );
              
              if (!shouldContinue) {
                navigate(redirectPath);
                return;
              }
            }
          }
        }

      } catch (error) {
        console.error('Session check failed:', error);
        // On error, redirect to be safe
        navigate(redirectPath);
      }
    };

    // Initial check
    checkSession();

    // Set up interval for periodic checks
    intervalRef.current = setInterval(checkSession, checkInterval);

    return () => {
      if (intervalRef.current) {
        clearInterval(intervalRef.current);
      }
    };
  }, [enabled, navigate, redirectPath, checkInterval]);

  // Cleanup on unmount
  useEffect(() => {
    return () => {
      if (intervalRef.current) {
        clearInterval(intervalRef.current);
      }
    };
  }, []);
};