import { useState, useEffect, useCallback, useRef } from 'react';

// Type declarations for Web Speech API
declare global {
  interface Window {
    SpeechRecognition: new () => SpeechRecognition;
    webkitSpeechRecognition: new () => SpeechRecognition;
  }
}

interface SpeechRecognition extends EventTarget {
  continuous: boolean;
  interimResults: boolean;
  lang: string;
  maxAlternatives: number;
  start(): void;
  stop(): void;
  abort(): void;
  onstart: ((this: SpeechRecognition, ev: Event) => any) | null;
  onend: ((this: SpeechRecognition, ev: Event) => any) | null;
  onerror: ((this: SpeechRecognition, ev: SpeechRecognitionErrorEvent) => any) | null;
  onresult: ((this: SpeechRecognition, ev: SpeechRecognitionEvent) => any) | null;
}

interface SpeechRecognitionEvent extends Event {
  results: SpeechRecognitionResultList;
  resultIndex: number;
}

interface SpeechRecognitionErrorEvent extends Event {
  error: string;
}

interface SpeechRecognitionResultList {
  readonly length: number;
  item(index: number): SpeechRecognitionResult;
  [index: number]: SpeechRecognitionResult;
}

interface SpeechRecognitionResult {
  readonly length: number;
  readonly isFinal: boolean;
  item(index: number): SpeechRecognitionAlternative;
  [index: number]: SpeechRecognitionAlternative;
}

interface SpeechRecognitionAlternative {
  readonly transcript: string;
  readonly confidence: number;
}

interface VoiceAssistantState {
  isListening: boolean;
  isSpeaking: boolean;
  isSupported: boolean;
  transcript: string;
  error: string | null;
}

interface VoiceAssistantHook extends VoiceAssistantState {
  speak: (text: string) => void;
  startListening: () => void;
  stopListening: () => void;
  stopSpeaking: () => void;
  resetTranscript: () => void;
  setLanguage: (lang: string) => void;
  language: string;
}

export const useVoiceAssistant = (): VoiceAssistantHook => {
  const [state, setState] = useState<VoiceAssistantState>({
    isListening: false,
    isSpeaking: false,
    isSupported: false,
    transcript: '',
    error: null,
  });

  const [language, setLanguageState] = useState<string>(() => {
    return localStorage.getItem('voiceLanguage') || 'en';
  });

  const recognitionRef = useRef<SpeechRecognition | null>(null);
  const synthRef = useRef<SpeechSynthesis | null>(null);
  const isInitialized = useRef(false);

  // Initialize APIs
  useEffect(() => {
    if (isInitialized.current) return;
    
    const SpeechRecognition = (window as any).SpeechRecognition || (window as any).webkitSpeechRecognition;
    const speechSynthesis = window.speechSynthesis;

    if (SpeechRecognition && speechSynthesis) {
      try {
        recognitionRef.current = new SpeechRecognition();
        synthRef.current = speechSynthesis;
        
        // Load voices (required for voice selection)
        if (speechSynthesis.getVoices().length === 0) {
          speechSynthesis.addEventListener('voiceschanged', () => {
            console.log('Voices loaded:', speechSynthesis.getVoices().length);
          });
        }
        
        // Optimal configuration for stability
        recognitionRef.current.continuous = false;
        recognitionRef.current.interimResults = false;
        recognitionRef.current.lang = language === 'kn' ? 'kn-IN' : 'en-US';
        recognitionRef.current.maxAlternatives = 1;

        // Setup event handlers
        recognitionRef.current.onstart = () => {
          console.log('Speech recognition started');
          setState(prev => ({ ...prev, isListening: true, error: null }));
        };

        recognitionRef.current.onresult = (event) => {
          const result = event.results[0];
          if (result && result[0]) {
            const transcript = result[0].transcript.trim();
            console.log('Speech recognized:', transcript);
            setState(prev => ({ ...prev, transcript }));
          }
        };

        recognitionRef.current.onerror = (event) => {
          console.error('Speech recognition error:', event.error);
          
          if (event.error === 'no-speech' || event.error === 'audio-capture') {
            // Ignore these common errors
            return;
          }
          
          setState(prev => ({ 
            ...prev, 
            isListening: false, 
            error: `Recognition error: ${event.error}` 
          }));
        };

        recognitionRef.current.onend = () => {
          console.log('Speech recognition ended');
          setState(prev => ({ ...prev, isListening: false }));
        };

        setState(prev => ({ ...prev, isSupported: true }));
        isInitialized.current = true;
      } catch (error) {
        console.error('Voice Assistant - Initialization error:', error);
        setState(prev => ({ ...prev, error: 'Failed to initialize speech recognition' }));
      }
    } else {
      setState(prev => ({ ...prev, error: 'Speech APIs not supported in this browser' }));
    }

    return () => {
      if (recognitionRef.current) {
        recognitionRef.current.abort();
      }
      if (synthRef.current) {
        synthRef.current.cancel();
      }
    };
  }, []);

  const speak = useCallback((text: string) => {
    if (!text.trim()) return;
    if (!synthRef.current) return;

    const speakWithVoice = () => {
      console.log('TTS: Language:', language, 'Text:', text);
      synthRef.current!.cancel();
      
      const utterance = new SpeechSynthesisUtterance(text);
      utterance.lang = language === 'kn' ? 'kn-IN' : 'en-US';
      
      const voices = synthRef.current!.getVoices();
      console.log('TTS: Total voices:', voices.length);
      
      if (language === 'kn') {
        const kannadaVoice = voices.find(v => v.lang === 'kn-IN' || v.lang.startsWith('kn') || v.name.toLowerCase().includes('kannada'));
        if (kannadaVoice) {
          utterance.voice = kannadaVoice;
          utterance.rate = 0.8;
          console.log('TTS: Using Kannada voice:', kannadaVoice.name);
        } else {
          const hindiVoice = voices.find(v => v.lang.startsWith('hi'));
          if (hindiVoice) {
            utterance.voice = hindiVoice;
            utterance.rate = 0.8;
            console.warn('TTS: No Kannada voice, using Hindi:', hindiVoice.name);
          } else {
            console.error('TTS: No Kannada/Hindi voice! Install Kannada language pack in Windows Settings.');
            console.error('TTS: Available voices:', voices.map(v => v.name + ' (' + v.lang + ')').join(', '));
            utterance.rate = 0.7;
          }
        }
      } else {
        const englishVoice = voices.find(v => v.lang === 'en-US' || v.lang.startsWith('en'));
        if (englishVoice) utterance.voice = englishVoice;
        utterance.rate = 0.9;
      }
      
      utterance.pitch = 1;
      utterance.volume = 1;

      utterance.onstart = () => {
        console.log('TTS: Speech synthesis STARTED');
        setState(prev => ({ ...prev, isSpeaking: true, error: null }));
      };
      
      utterance.onend = () => {
        console.log('TTS: Speech synthesis ENDED');
        setState(prev => ({ ...prev, isSpeaking: false }));
      };
      
      utterance.onerror = (event) => {
        if (event.error === 'interrupted' || event.error === 'canceled') {
          console.log('TTS: Speech stopped by user');
          setState(prev => ({ ...prev, isSpeaking: false }));
          return;
        }
        console.error('TTS: Speech synthesis ERROR:', event.error);
        setState(prev => ({ ...prev, isSpeaking: false, error: `Speech error: ${event.error}` }));
      };

      try {
        synthRef.current!.speak(utterance);
        console.log('TTS: speechSynthesis.speak() called');
      } catch (error) {
        console.error('TTS: Error calling speak():', error);
        setState(prev => ({ ...prev, error: `Failed to start speech: ${error}` }));
      }
    };

    // Wait for voices to load if not ready
    const voices = synthRef.current.getVoices();
    if (voices.length === 0) {
      console.log('TTS: Waiting for voices to load...');
      const handler = () => {
        console.log('TTS: Voices loaded, speaking now');
        speakWithVoice();
        synthRef.current!.removeEventListener('voiceschanged', handler);
      };
      synthRef.current.addEventListener('voiceschanged', handler);
    } else {
      speakWithVoice();
    }
  }, [language]);



  const startListening = useCallback(async () => {
    if (!recognitionRef.current || state.isListening) return;

    // Don't start listening if TTS is currently speaking
    if (state.isSpeaking) {
      console.log('STT: Waiting for TTS to finish before starting recognition');
      return;
    }

    try {
      // Request microphone permission
      const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
      stream.getTracks().forEach(track => track.stop());
      
      console.log('Starting speech recognition - single session');
      setState(prev => ({ ...prev, transcript: '', error: null }));
      recognitionRef.current.start();
    } catch (error: any) {
      const errorMessage = error.name === 'NotAllowedError' 
        ? 'Microphone permission denied. Please allow microphone access and try again.'
        : `Failed to start listening: ${error.message || error}`;
      setState(prev => ({ ...prev, error: errorMessage }));
    }
  }, [state.isListening, state.isSpeaking]);

  const stopListening = useCallback(() => {
    if (recognitionRef.current && state.isListening) {
      recognitionRef.current.stop();
    }
  }, [state.isListening]);

  const stopSpeaking = useCallback(() => {
    if (synthRef.current) {
      synthRef.current.cancel();
      setState(prev => ({ ...prev, isSpeaking: false }));
    }
  }, []);

  const resetTranscript = useCallback(() => {
    setState(prev => ({ ...prev, transcript: '' }));
  }, []);

  const setLanguage = useCallback((lang: string) => {
    setLanguageState(lang);
    localStorage.setItem('voiceLanguage', lang);
    
    // Update recognition language if initialized
    if (recognitionRef.current) {
      recognitionRef.current.lang = lang === 'kn' ? 'kn-IN' : 'en-US';
    }
  }, []);

  return {
    ...state,
    speak,
    startListening,
    stopListening,
    stopSpeaking,
    resetTranscript,
    setLanguage,
    language,
  };
};