import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import axios from 'axios';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8000';

interface CreditPackage {
  id: number;
  name: string;
  credits: number;
  price: number;
  currency: string;
  bonus_credits: number;
  discount_percent: number;
  is_active: boolean;
}

interface WalletBalance {
  balance: number;
  total_earned: number;
  total_spent: number;
  total_purchased: number;
  last_transaction_at: string | null;
}

interface Service {
  id: number;
  service_code: string;
  service_name: string;
  service_category: string;
  credits_required: number;
  description: string;
  is_active: boolean;
}

export const useWalletBalance = (userId: number, userType: string) => {
  return useQuery<WalletBalance>({
    queryKey: ['wallet', 'balance', userId, userType],
    queryFn: async () => {
      const { data } = await axios.get(`${API_URL}/api/wallet/balance`, {
        params: { user_id: userId, user_type: userType }
      });
      return data;
    }
  });
};

export const useCreditPackages = (audienceRole?: string) => {
  return useQuery<CreditPackage[]>({
    queryKey: ['wallet', 'packages', audienceRole],
    queryFn: async () => {
      const { data } = await axios.get(`${API_URL}/api/wallet/packages`, {
        params: audienceRole ? { audience_role: audienceRole } : {}
      });
      return data;
    }
  });
};

export const usePurchaseCredits = (userId: number, userType: string) => {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async (packageId: number) => {
      const { data } = await axios.post(
        `${API_URL}/api/wallet/purchase`,
        { package_id: packageId, payment_method: 'razorpay' },
        { params: { user_id: userId, user_type: userType } }
      );
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['wallet', 'balance'] });
      queryClient.invalidateQueries({ queryKey: ['wallet', 'transactions'] });
    }
  });
};

export const useTransactionHistory = (userId: number, limit = 50) => {
  return useQuery({
    queryKey: ['wallet', 'transactions', userId, limit],
    queryFn: async () => {
      const { data } = await axios.get(`${API_URL}/api/wallet/transactions`, {
        params: { user_id: userId, limit }
      });
      return data.data;
    }
  });
};

export const useServices = (category?: string) => {
  return useQuery<Service[]>({
    queryKey: ['wallet', 'services', category],
    queryFn: async () => {
      const { data } = await axios.get(`${API_URL}/api/wallet/services`, {
        params: category ? { category } : {}
      });
      return data.data;
    }
  });
};

export const useServiceUsage = (userId: number, userType: string) => {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async ({ serviceCode, referenceType, referenceId }: {
      serviceCode: string;
      referenceType?: string;
      referenceId?: number;
    }) => {
      const { data } = await axios.post(
        `${API_URL}/api/wallet/use-service`,
        { service_code: serviceCode, reference_type: referenceType, reference_id: referenceId },
        { params: { user_id: userId, user_type: userType } }
      );
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['wallet', 'balance'] });
      queryClient.invalidateQueries({ queryKey: ['wallet', 'transactions'] });
      queryClient.invalidateQueries({ queryKey: ['wallet', 'usage'] });
    }
  });
};

export const useUsageHistory = (userId: number, limit = 50) => {
  return useQuery({
    queryKey: ['wallet', 'usage', userId, limit],
    queryFn: async () => {
      const { data } = await axios.get(`${API_URL}/api/wallet/usage-history`, {
        params: { user_id: userId, limit }
      });
      return data.data;
    }
  });
};

export const useAdminStats = () => {
  return useQuery({
    queryKey: ['wallet', 'admin', 'stats'],
    queryFn: async () => {
      const { data } = await axios.get(`${API_URL}/api/wallet/admin/stats`);
      return data.data;
    }
  });
};

export const useCreatePackage = () => {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async (packageData: any) => {
      const { data } = await axios.post(`${API_URL}/api/wallet/admin/packages`, packageData);
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['wallet', 'packages'] });
    }
  });
};

export const useTogglePackage = () => {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async (packageId: number) => {
      const { data } = await axios.patch(`${API_URL}/api/wallet/admin/packages/${packageId}/toggle`);
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['wallet', 'packages'] });
    }
  });
};
