import { useState, useCallback, useRef, useEffect } from 'react';

interface CallState {
  status: 'idle' | 'calling' | 'ringing' | 'connected' | 'ended' | 'rejected';
  error: string | null;
  duration: number;
  isAudioEnabled: boolean;
  pendingOffer?: string;
  pendingIceCandidates?: RTCIceCandidate[];
}

export const useWebRTCCall = (negotiationId: string, userId: string) => {
  const [callState, setCallState] = useState<CallState>({
    status: 'idle',
    error: null,
    duration: 0,
    isAudioEnabled: true,
  });
  
  // Use ref to access current state in callbacks
  const callStateRef = useRef(callState);
  useEffect(() => {
    callStateRef.current = callState;
  }, [callState]);

  const peerConnection = useRef<RTCPeerConnection | null>(null);
  const localStream = useRef<MediaStream | null>(null);
  const remoteAudio = useRef<HTMLAudioElement | null>(null);
  const websocket = useRef<WebSocket | null>(null);
  const durationInterval = useRef<NodeJS.Timeout | null>(null);

  // Initialize WebRTC peer connection
  const initializePeerConnection = useCallback(() => {
    const config: RTCConfiguration = {
      iceServers: [
        { urls: 'stun:stun.l.google.com:19302' },
        { urls: 'stun:stun1.l.google.com:19302' },
        // Free TURN servers for better connectivity
        { 
          urls: 'turn:openrelay.metered.ca:80',
          username: 'openrelayproject',
          credential: 'openrelayproject'
        },
        {
          urls: 'turn:openrelay.metered.ca:443',
          username: 'openrelayproject', 
          credential: 'openrelayproject'
        }
      ],
    };

    peerConnection.current = new RTCPeerConnection(config);

    // Handle remote stream
    peerConnection.current.ontrack = (event) => {
      console.log('Received remote stream:', event.streams[0]);
      if (remoteAudio.current) {
        remoteAudio.current.srcObject = event.streams[0];
        // Set maximum volume and ensure not muted
        remoteAudio.current.volume = 1.0;
        remoteAudio.current.muted = false;
        console.log('Set remote audio source');
        console.log('Remote audio element:', remoteAudio.current);
        console.log('Remote audio volume:', remoteAudio.current.volume);
        console.log('Remote audio muted:', remoteAudio.current.muted);
        remoteAudio.current.play().then(() => {
          console.log('Remote audio playing successfully');
        }).catch((error) => {
          console.error('Failed to play remote audio:', error);
        });
      }
    };

    // Handle ICE candidates
    peerConnection.current.onicecandidate = (event) => {
      if (event.candidate && websocket.current) {
        websocket.current.send(JSON.stringify({
          type: 'ice-candidate',
          candidate: event.candidate,
        }));
      }
    };

    // Handle connection state changes with ruthless monitoring
    peerConnection.current.onconnectionstatechange = () => {
      const state = peerConnection.current?.connectionState;
      console.log('Peer connection state:', state);
      if (state === 'connected') {
        setCallState(prev => ({ ...prev, status: 'connected' }));
        startDurationTimer();
        // Monitor connection quality
        monitorConnectionQuality();
      } else if (state === 'disconnected' || state === 'failed') {
        console.log('Peer connection failed/disconnected');
        setCallState(prev => ({ ...prev, error: 'Connection lost' }));
        endCall();
      }
    };
  }, []);

  // Get authentication token from any dashboard
  const getAuthToken = useCallback(() => {
    return localStorage.getItem('access_token') || sessionStorage.getItem('access_token') ||
           localStorage.getItem('sellerToken') || sessionStorage.getItem('sellerToken') ||
           localStorage.getItem('agentToken') || sessionStorage.getItem('agentToken');
  }, []);

  // Initialize WebSocket connection
  const initializeWebSocket = useCallback(() => {
    // Don't create new connection if one already exists and is open
    if (websocket.current && websocket.current.readyState === WebSocket.OPEN) {
      return;
    }
    
    // Close existing connection first
    if (websocket.current && websocket.current.readyState !== WebSocket.CLOSED) {
      websocket.current.close();
    }
    
    const token = getAuthToken();
    // Use http/https protocol for WebSocket to match CSP
    const protocol = window.location.protocol === 'https:' ? 'wss:' : 'ws:';
    const baseUrl = import.meta.env.VITE_API_URL || 'http://localhost:8090/api';
    // Extract host and port from API URL
    const apiUrl = new URL(baseUrl.replace('/api', ''));
    const wsUrl = `${protocol}//${apiUrl.host}/api/negotiations/ws/${negotiationId}?token=${token}`;
    
    console.log(`Connecting to '${wsUrl}'`);
    websocket.current = new WebSocket(wsUrl);

    websocket.current.onopen = () => {
      console.log('WebSocket connected successfully');
    };

    websocket.current.onmessage = async (event) => {
      const message = JSON.parse(event.data);
      console.log('WebSocket message received:', message);

      switch (message.type) {
        case 'incoming_call':
          console.log('Incoming call received, setting status to ringing');
          setCallState(prev => ({ ...prev, status: 'ringing' }));
          // Initialize peer connection immediately when call comes in
          if (!peerConnection.current) {
            console.log('Initializing peer connection for incoming call');
            initializePeerConnection();
          }
          break;

        case 'call_accepted':
          setCallState(prev => ({ ...prev, status: 'connected' }));
          break;

        case 'call_rejected':
          setCallState(prev => ({ ...prev, status: 'rejected', error: message.reason }));
          cleanup();
          break;

        case 'call_ended':
          setCallState(prev => ({ ...prev, status: 'ended' }));
          cleanup();
          break;

        case 'offer':
          // Only process offer if user has accepted the call
          if (callState.status === 'connected') {
            console.log('Processing WebRTC offer after user accepted');
            if (peerConnection.current) {
              await peerConnection.current.setRemoteDescription(new RTCSessionDescription({
                type: 'offer',
                sdp: message.sdp,
              }));
              
              const answer = await peerConnection.current.createAnswer();
              await peerConnection.current.setLocalDescription(answer);
              
              websocket.current?.send(JSON.stringify({
                type: 'answer',
                sdp: answer.sdp,
              }));
            }
          } else {
            console.log('Received offer but call not accepted yet, storing for later');
            // Store the offer to process after user accepts
            setCallState(prev => ({ ...prev, pendingOffer: message.sdp }));
          }
          break;

        case 'answer':
          if (peerConnection.current) {
            await peerConnection.current.setRemoteDescription(new RTCSessionDescription({
              type: 'answer',
              sdp: message.sdp,
            }));
          }
          break;

        case 'ice-candidate':
          if (peerConnection.current && peerConnection.current.remoteDescription) {
            await peerConnection.current.addIceCandidate(new RTCIceCandidate(message.candidate));
          } else {
            // Queue ICE candidates until remote description is set
            console.log('Queueing ICE candidate until remote description is ready');
            setCallState(prev => ({
              ...prev,
              pendingIceCandidates: [...(prev.pendingIceCandidates || []), new RTCIceCandidate(message.candidate)]
            }));
          }
          break;
      }
    };

    websocket.current.onerror = (error) => {
      console.error('WebSocket error:', error);
      setCallState(prev => ({ ...prev, error: 'Connection failed' }));
    };

    websocket.current.onclose = () => {
      console.log('WebSocket disconnected');
      cleanup();
    };
  }, [negotiationId]);

  // Start duration timer
  const startDurationTimer = useCallback(() => {
    const startTime = Date.now();
    durationInterval.current = setInterval(() => {
      setCallState(prev => ({
        ...prev,
        duration: Math.floor((Date.now() - startTime) / 1000),
      }));
    }, 1000);
  }, []);

  // Get user media
  const getUserMedia = useCallback(async () => {
    try {
      const stream = await navigator.mediaDevices.getUserMedia({ 
        audio: true, 
        video: false 
      });
      localStream.current = stream;
      
      // Add audio track to peer connection
      if (peerConnection.current) {
        stream.getTracks().forEach(track => {
          console.log('Adding track to peer connection:', track);
          peerConnection.current?.addTrack(track, stream);
        });
      }
      
      return stream;
    } catch (error) {
      console.error('Failed to get user media:', error);
      setCallState(prev => ({ 
        ...prev, 
        error: 'Microphone access denied. Please allow microphone access and try again.' 
      }));
      throw error;
    }
  }, []);

  // Start call
  const startCall = useCallback(async () => {
    try {
      // Prevent starting call if already in progress
      if (callState.status !== 'idle') {
        console.log('Call already in progress, status:', callState.status);
        return;
      }
      
      setCallState(prev => ({ ...prev, status: 'calling', error: null }));
      
      // Call backend API first
      const apiUrl = import.meta.env.VITE_API_URL || 'http://localhost:8090/api';
      console.log('Using API URL:', apiUrl);
      console.log('Making call to:', `${apiUrl}/negotiations/${negotiationId}/call`);
      const response = await fetch(`${apiUrl}/negotiations/${negotiationId}/call`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${getAuthToken()}`,
        },
        body: JSON.stringify({ action: 'call_start' }),
      });

      if (!response.ok) {
        const error = await response.json();
        console.error('Start call API error:', error);
        throw new Error(error.detail || 'Failed to start call');
      }
      
      console.log('Call started successfully on backend');
      // Start duration timer immediately for caller
      startDurationTimer();
      
      // Initialize connections after successful API call (only if not already connected)
      if (!websocket.current || websocket.current.readyState !== WebSocket.OPEN) {
        initializeWebSocket();
      }
      if (!peerConnection.current) {
        console.log('Initializing peer connection for caller');
        initializePeerConnection();
      }
      
      // Get user media
      await getUserMedia();
      
      // Create offer
      if (peerConnection.current) {
        const offer = await peerConnection.current.createOffer();
        await peerConnection.current.setLocalDescription(offer);
        
        // Send offer via WebSocket when ready
        if (websocket.current && websocket.current.readyState === WebSocket.OPEN) {
          websocket.current.send(JSON.stringify({
            type: 'offer',
            sdp: offer.sdp,
          }));
        }
      }

    } catch (error: any) {
      console.error('Failed to start call:', error);
      setCallState(prev => ({ 
        ...prev, 
        status: 'idle', 
        error: error.message || 'Failed to start call' 
      }));
      cleanup();
    }
  }, [negotiationId, initializePeerConnection, initializeWebSocket, getUserMedia]);

  // Accept call
  const acceptCall = useCallback(async () => {
    try {
      // Don't initialize peer connection here, just accept the call
      const apiUrl = import.meta.env.VITE_API_URL || 'http://localhost:8090/api';
      const response = await fetch(`${apiUrl}/negotiations/${negotiationId}/call`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${getAuthToken()}`,
        },
        body: JSON.stringify({ action: 'call_accept' }),
      });

      if (!response.ok) {
        const error = await response.json();
        console.error('Accept call API error:', error);
        throw new Error(error.detail || 'Failed to accept call');
      }
      
      console.log('Call accepted successfully on backend');
      // Set state to connected immediately after successful API call
      setCallState(prev => ({ ...prev, status: 'connected' }));
      // Start duration timer immediately
      startDurationTimer();

      // Initialize peer connection and get media after successful API call
      initializePeerConnection();
      await getUserMedia();
      
      // Use current state values from ref instead of stale closure
      const currentState = callStateRef.current;
      console.log('Current state:', currentState);
      console.log('Pending offer exists:', !!currentState.pendingOffer);
      console.log('Pending ICE candidates count:', currentState.pendingIceCandidates?.length || 0);
      
      // Process pending offer if available
      if (currentState.pendingOffer && peerConnection.current) {
        console.log('Processing pending WebRTC offer');
        await peerConnection.current.setRemoteDescription(new RTCSessionDescription({
          type: 'offer',
          sdp: currentState.pendingOffer,
        }));
        
        const answer = await peerConnection.current.createAnswer();
        await peerConnection.current.setLocalDescription(answer);
        
        websocket.current?.send(JSON.stringify({
          type: 'answer',
          sdp: answer.sdp,
        }));
        
        // Process any pending ICE candidates
        if (currentState.pendingIceCandidates) {
          console.log('Processing', currentState.pendingIceCandidates.length, 'pending ICE candidates');
          for (const candidate of currentState.pendingIceCandidates) {
            await peerConnection.current.addIceCandidate(candidate);
          }
        }
        
        // Clear pending data
        setCallState(prev => ({ ...prev, pendingOffer: undefined, pendingIceCandidates: undefined }));
      } else {
        console.log('No pending offer found:', currentState.pendingOffer);
      }

    } catch (error: any) {
      console.error('Failed to accept call:', error);
      setCallState(prev => ({ ...prev, error: error.message }));
    }
  }, [negotiationId, initializePeerConnection, getUserMedia]);

  // Reject call
  const rejectCall = useCallback(async (reason?: string) => {
    try {
      const apiUrl = import.meta.env.VITE_API_URL || 'http://localhost:8090/api';
      const response = await fetch(`${apiUrl}/negotiations/${negotiationId}/call`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${getAuthToken()}`,
        },
        body: JSON.stringify({ action: 'call_reject', reason }),
      });

      if (response.ok) {
        setCallState(prev => ({ ...prev, status: 'rejected' }));
      }
    } catch (error) {
      console.error('Failed to reject call:', error);
    }
    cleanup();
  }, [negotiationId]);

  // End call
  const endCall = useCallback(async () => {
    try {
      const apiUrl = import.meta.env.VITE_API_URL || 'http://localhost:8090/api';
      const response = await fetch(`${apiUrl}/negotiations/${negotiationId}/call`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${getAuthToken()}`,
        },
        body: JSON.stringify({ action: 'call_end' }),
      });

      if (response.ok) {
        setCallState(prev => ({ ...prev, status: 'ended' }));
      }
    } catch (error) {
      console.error('Failed to end call:', error);
    }
    cleanup();
  }, [negotiationId]);

  // Toggle audio
  const toggleAudio = useCallback(() => {
    if (localStream.current) {
      const audioTrack = localStream.current.getAudioTracks()[0];
      if (audioTrack) {
        audioTrack.enabled = !audioTrack.enabled;
        setCallState(prev => ({ ...prev, isAudioEnabled: audioTrack.enabled }));
      }
    }
  }, []);

  // Monitor connection quality ruthlessly
  const monitorConnectionQuality = useCallback(() => {
    if (!peerConnection.current) return;
    
    const checkQuality = async () => {
      try {
        const stats = await peerConnection.current!.getStats();
        let packetsLost = 0;
        let packetsReceived = 0;
        
        stats.forEach(report => {
          if (report.type === 'inbound-rtp' && report.kind === 'audio') {
            packetsLost += report.packetsLost || 0;
            packetsReceived += report.packetsReceived || 0;
          }
        });
        
        const lossRate = packetsReceived > 0 ? (packetsLost / packetsReceived) * 100 : 0;
        
        // Ruthless quality enforcement: end call if quality is poor
        if (lossRate > 10) {
          console.warn(`Poor connection quality: ${lossRate.toFixed(2)}% packet loss`);
          setCallState(prev => ({ ...prev, error: 'Poor connection quality detected' }));
          endCall();
        }
      } catch (error) {
        console.error('Failed to check connection quality:', error);
      }
    };
    
    // Check quality every 10 seconds
    const qualityInterval = setInterval(checkQuality, 10000);
    
    // Clear interval when call ends
    return () => clearInterval(qualityInterval);
  }, [endCall]);

  // Cleanup function
  const cleanup = useCallback(() => {
    // Stop duration timer
    if (durationInterval.current) {
      clearInterval(durationInterval.current);
      durationInterval.current = null;
    }

    // Close peer connection
    if (peerConnection.current) {
      peerConnection.current.close();
      peerConnection.current = null;
    }

    // Stop local stream
    if (localStream.current) {
      localStream.current.getTracks().forEach(track => track.stop());
      localStream.current = null;
    }

    // Close WebSocket
    if (websocket.current) {
      websocket.current.close();
      websocket.current = null;
    }

    // Stop remote audio
    if (remoteAudio.current) {
      remoteAudio.current.srcObject = null;
    }
  }, []);

  // Cleanup on unmount
  useEffect(() => {
    return cleanup;
  }, [cleanup]);

  // Initialize WebSocket on mount for incoming calls
  useEffect(() => {
    initializeWebSocket();
    return () => {
      if (websocket.current && websocket.current.readyState !== WebSocket.CLOSED) {
        websocket.current.close();
      }
    };
  }, [negotiationId]); // Only re-initialize if negotiationId changes

  return {
    callState,
    startCall,
    acceptCall,
    rejectCall,
    endCall,
    toggleAudio,
    remoteAudio,
  };
};