import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { escrowApi } from '@/api/escrow';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { toast } from '@/utils/toast';

export const AdminEscrowList: React.FC = () => {
  const navigate = useNavigate();
  const [escrows, setEscrows] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    loadEscrows();
  }, []);

  const loadEscrows = async () => {
    setLoading(true);
    setError(null);
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/escrow/?all_escrows=true', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      
      if (response.status === 404) {
        // Endpoint not available - show empty state
        setEscrows([]);
        return;
      }
      
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`HTTP ${response.status}: ${errorText}`);
      }
      
      const data = await response.json();
      setEscrows(data.filter((e: any) => e.status === 'funded'));
    } catch (error: any) {
      console.error('Failed to load escrows:', error);
      setError(error.message || 'Failed to load escrows');
      setEscrows([]);
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    return 'bg-yellow-100 text-yellow-800';
  };

  if (loading) return <div className="p-8">Loading...</div>;

  if (error) {
    return (
      <div className="max-w-6xl mx-auto p-8">
        <Card className="p-8 text-center">
          <p className="text-red-600 mb-4">Error: {error}</p>
          <Button onClick={loadEscrows}>Retry</Button>
        </Card>
      </div>
    );
  }

  return (
    <div className="max-w-6xl mx-auto p-8">
      <h1 className="text-3xl font-bold mb-6">Escrow Review Queue</h1>
      
      {escrows.length === 0 ? (
        <Card className="p-8 text-center">
          <p className="text-gray-600">No escrows pending review</p>
        </Card>
      ) : (
        <div className="grid gap-4">
          {escrows.map((escrow) => (
            <Card key={escrow.id} className="p-6">
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="text-xl font-bold">{escrow.escrow_number}</h3>
                    <Badge className={getStatusColor(escrow.status)}>
                      PENDING REVIEW
                    </Badge>
                  </div>
                  <div className="grid grid-cols-3 gap-4 mt-4">
                    <div>
                      <label className="text-sm text-gray-600">Amount</label>
                      <p className="text-lg font-semibold text-green-600">
                        ₹{escrow.escrow_amount?.toLocaleString()}
                      </p>
                    </div>
                    <div>
                      <label className="text-sm text-gray-600">Property ID</label>
                      <p className="font-semibold">#{escrow.property_id}</p>
                    </div>
                    <div>
                      <label className="text-sm text-gray-600">Created</label>
                      <p className="text-sm">
                        {new Date(escrow.created_at).toLocaleDateString()}
                      </p>
                    </div>
                  </div>
                </div>
                <Button
                  onClick={() => navigate(`/admin/escrow/${escrow.id}`)}
                  className="ml-4"
                >
                  Review →
                </Button>
              </div>
            </Card>
          ))}
        </div>
      )}
    </div>
  );
};
