import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { escrowApi } from '@/api/escrow';
import { agreementsApi } from '@/api/agreements';
import { contractsApi } from '@/api/contracts';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { toast } from '@/utils/toast';
import { Textarea } from '@/components/ui/textarea';

export const AdminEscrowReview: React.FC = () => {
  const { id } = useParams<{ id: string }>();
  const navigate = useNavigate();
  const [escrow, setEscrow] = useState<any>(null);
  const [contract, setContract] = useState<any>(null);
  const [agreement, setAgreement] = useState<any>(null);
  const [signatures, setSignatures] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [rejectionReason, setRejectionReason] = useState('');
  const [showRejectModal, setShowRejectModal] = useState(false);
  const [processing, setProcessing] = useState(false);

  useEffect(() => {
    loadData();
  }, [id]);

  const loadData = async () => {
    try {
      const escrowData = await escrowApi.getById(id!);
      setEscrow(escrowData);

      if (escrowData.contract_id) {
        const contractData = await contractsApi.getById(escrowData.contract_id);
        setContract(contractData);

        const sigs = await contractsApi.getSignatures(escrowData.contract_id);
        setSignatures(sigs);

        if (contractData.agreement_id) {
          const agreementData = await agreementsApi.getById(contractData.agreement_id);
          setAgreement(agreementData);
        }
      }
    } catch (error) {
      toast.error('Failed to load escrow details');
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async () => {
    if (!confirm('Are you sure you want to approve this transaction? Funds will be released to the seller.')) return;
    
    setProcessing(true);
    try {
      await escrowApi.approve(id!);
      toast.success('Transaction approved! Funds released to seller.');
      navigate('/admin/dashboard', { state: { activeSection: 'EscrowReview' } });
    } catch (error) {
      toast.error('Failed to approve transaction');
    } finally {
      setProcessing(false);
    }
  };

  const handleReject = async () => {
    if (!rejectionReason.trim()) {
      toast.error('Please provide a rejection reason');
      return;
    }

    setProcessing(true);
    try {
      await escrowApi.reject(id!, rejectionReason);
      toast.success('Transaction rejected. Funds refunded to buyer.');
      navigate('/admin/dashboard', { state: { activeSection: 'EscrowReview' } });
    } catch (error) {
      toast.error('Failed to reject transaction');
    } finally {
      setProcessing(false);
      setShowRejectModal(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!escrow) {
    return <div className="p-8 text-center text-gray-500">Escrow not found</div>;
  }

  return (
    <div className="min-h-screen bg-gray-100 py-8 px-4">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="bg-white rounded-lg shadow-md p-6 mb-6">
          <div className="flex justify-between items-center">
            <div>
              <h1 className="text-3xl font-bold text-gray-900">Escrow Transaction Review</h1>
              <p className="text-gray-600 mt-1">Escrow #{escrow.escrow_number}</p>
            </div>
            <Badge className="bg-red-600 text-white px-4 py-2">PENDING ADMIN APPROVAL</Badge>
          </div>
        </div>

        {/* Escrow Amount */}
        <div className="bg-white rounded-lg shadow-lg p-8 mb-6 border-4 border-green-600">
          <div className="text-center">
            <p className="text-xl font-semibold mb-3 text-gray-700">Escrow Amount</p>
            <p className="text-6xl font-bold mt-2 mb-3 text-gray-900">₹{escrow.escrow_amount?.toLocaleString()}</p>
            <div className="inline-block bg-green-600 text-white px-6 py-2 rounded-full font-bold text-base">
              Status: {escrow.status.toUpperCase()}
            </div>
          </div>
        </div>

        {/* Agreement Section */}
        {agreement && (
          <div className="bg-white rounded-lg shadow-md p-6 mb-6">
            <div className="border-b border-gray-200 pb-4 mb-4">
              <h2 className="text-2xl font-bold text-gray-900">📄 Agreement Document</h2>
              <div className="flex gap-4 mt-2 text-sm">
                <span className="text-gray-600">Agreement #: <span className="font-semibold text-gray-900">{agreement.agreement_number}</span></span>
                <span className="text-gray-600">Type: <span className="font-semibold text-gray-900 capitalize">{agreement.agreement_type?.replace('_', ' ')}</span></span>
                <Badge className="bg-green-600 text-white">{agreement.agreement_status}</Badge>
              </div>
            </div>
            <div className="bg-gray-50 border-2 border-gray-300 rounded-lg p-6">
              <div className="prose max-w-none">
                <div className="text-gray-800 leading-relaxed whitespace-pre-wrap" style={{ fontFamily: 'Georgia, serif' }}>
                  {agreement.content || 'No agreement content available'}
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Contract Section */}
        {contract && (
          <div className="bg-white rounded-lg shadow-md p-6 mb-6">
            <div className="border-b border-gray-200 pb-4 mb-4">
              <h2 className="text-2xl font-bold text-gray-900">📋 Contract Terms</h2>
              <div className="flex gap-4 mt-2 text-sm">
                <span className="text-gray-600">Contract #: <span className="font-semibold text-gray-900">{contract.contract_number}</span></span>
                <Badge className="bg-blue-600 text-white">{contract.status}</Badge>
              </div>
            </div>
            <div className="bg-gray-50 border-2 border-gray-300 rounded-lg p-6">
              <div className="prose max-w-none">
                <div className="text-gray-800 leading-relaxed whitespace-pre-wrap" style={{ fontFamily: 'Georgia, serif' }}>
                  {contract.contract_terms || 'No contract terms available'}
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Signatures Section */}
        <div className="bg-white rounded-lg shadow-md p-6 mb-6">
          <div className="border-b border-gray-200 pb-4 mb-4">
            <h2 className="text-2xl font-bold text-gray-900">✍️ Digital Signatures</h2>
            <p className="text-gray-600 text-sm mt-1">{signatures.length} signatures collected</p>
          </div>
          
          {signatures.length > 0 ? (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {signatures.map((sig, index) => (
                <div key={sig.id} className="border-2 border-green-500 bg-green-50 rounded-lg p-4">
                  <div className="flex items-start gap-3">
                    <div className="bg-green-600 text-white rounded-full w-10 h-10 flex items-center justify-center font-bold text-lg flex-shrink-0">
                      ✓
                    </div>
                    <div className="flex-1">
                      <p className="font-bold text-gray-900 text-lg">{sig.signer_name || `Party ${index + 1}`}</p>
                      <p className="text-sm text-gray-700 font-medium capitalize">{sig.signer_role || 'Signer'}</p>
                      <p className="text-xs text-gray-600 mt-2">Signed on: {new Date(sig.signed_at).toLocaleString()}</p>
                      {sig.signature_data && (
                        <div className="mt-3 bg-white border border-gray-300 rounded p-2">
                          <img src={sig.signature_data} alt="Signature" className="h-16 w-auto" />
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center py-8 bg-gray-50 rounded-lg">
              <p className="text-gray-500">No signatures available</p>
            </div>
          )}
        </div>

        {/* Admin Actions */}
        {escrow.status === 'funded' && (
          <div className="bg-white rounded-lg shadow-md p-6 border-2 border-blue-500">
            <h2 className="text-2xl font-bold text-gray-900 mb-4">⚖️ Admin Decision</h2>
            <p className="text-gray-700 mb-6">
              Review all documents and signatures above. Approve to release funds to seller, or reject to refund buyer.
            </p>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <button
                onClick={handleApprove}
                disabled={processing}
                className="bg-green-600 hover:bg-green-700 text-black font-bold py-4 px-6 rounded-lg text-lg transition-colors disabled:opacity-50"
              >
                {processing ? 'Processing...' : 'Approve & Release Funds'}
              </button>
              <button
                onClick={() => setShowRejectModal(true)}
                disabled={processing}
                className="bg-red-600 hover:bg-red-700 text-white font-bold py-4 px-6 rounded-lg text-lg transition-colors disabled:opacity-50"
              >
                Reject & Refund Buyer
              </button>
            </div>
          </div>
        )}
      </div>

      {/* Reject Modal */}
      {showRejectModal && (
        <div className="fixed inset-0 bg-black bg-opacity-60 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg shadow-xl p-6 max-w-md w-full">
            <h3 className="text-2xl font-bold mb-4 text-gray-900">Reject Transaction</h3>
            <p className="text-gray-600 mb-4">
              Provide a reason for rejection. The buyer will be refunded.
            </p>
            <Textarea
              value={rejectionReason}
              onChange={(e) => setRejectionReason(e.target.value)}
              placeholder="Enter rejection reason..."
              className="mb-4 border-2 border-gray-300"
              rows={4}
            />
            <div className="flex gap-3">
              <button
                onClick={() => setShowRejectModal(false)}
                className="flex-1 bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-3 px-4 rounded-lg transition-colors"
                disabled={processing}
              >
                Cancel
              </button>
              <button
                onClick={handleReject}
                className="flex-1 bg-red-600 hover:bg-red-700 text-white font-bold py-3 px-4 rounded-lg transition-colors disabled:opacity-50"
                disabled={processing || !rejectionReason.trim()}
              >
                {processing ? 'Processing...' : 'Confirm Rejection'}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};
