import React, { useState, useEffect } from 'react';
import axios from 'axios';

interface Ticket {
  id: number;
  ticket_id: string;
  user_id: number;
  user_name: string;
  user_type: string;
  category: string;
  priority: string;
  status: string;
  subject: string;
  property_id?: number;
  created_at: string;
  responses_count: number;
}

interface TicketDetail {
  id: number;
  ticket_id: string;
  user_id: number;
  user_name: string;
  user_type: string;
  category: string;
  priority: string;
  status: string;
  subject: string;
  description: string;
  property_id?: number;
  created_at: string;
  responses: Array<{
    id: number;
    responder_name: string;
    responder_type: string;
    message: string;
    created_at: string;
  }>;
  attachments: Array<{
    id: number;
    file_name: string;
    file_path: string;
  }>;
}

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8000';

const AdminSupport: React.FC = () => {
  const [tickets, setTickets] = useState<Ticket[]>([]);
  const [selectedTicket, setSelectedTicket] = useState<TicketDetail | null>(null);
  const [responseMessage, setResponseMessage] = useState('');
  const [loading, setLoading] = useState(false);

  const adminId = parseInt(localStorage.getItem('user_id') || sessionStorage.getItem('user_id') || '0');

  useEffect(() => {
    fetchAllTickets();
  }, []);

  const fetchAllTickets = async () => {
    try {
      const response = await axios.get(`${API_URL}/support/tickets`, {
        params: { is_admin: true }
      });
      setTickets(response.data);
    } catch (error) {
      console.error('Error fetching tickets:', error);
    }
  };

  const viewTicketDetails = async (ticketId: string) => {
    try {
      const response = await axios.get(`${API_URL}/support/tickets/${ticketId}`);
      setSelectedTicket(response.data);
      setResponseMessage('');
    } catch (error) {
      console.error('Error fetching ticket details:', error);
    }
  };

  const updateStatus = async (status: string) => {
    if (!selectedTicket) return;

    try {
      await axios.post(`${API_URL}/support/tickets/status`, {
        ticket_id: selectedTicket.id,
        status: status
      });
      alert('Status updated successfully');
      fetchAllTickets();
      viewTicketDetails(selectedTicket.ticket_id);
    } catch (error) {
      console.error('Error updating status:', error);
      alert('Failed to update status');
    }
  };

  const sendResponse = async () => {
    if (!selectedTicket || !responseMessage.trim()) {
      alert('Please enter a response message');
      return;
    }

    setLoading(true);
    try {
      await axios.post(`${API_URL}/support/tickets/response`, {
        ticket_id: selectedTicket.id,
        responder_id: adminId,
        responder_type: 'admin',
        message: responseMessage
      });
      alert('Response sent successfully');
      setResponseMessage('');
      viewTicketDetails(selectedTicket.ticket_id);
      fetchAllTickets();
    } catch (error) {
      console.error('Error sending response:', error);
      alert('Failed to send response');
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      open: '#3b82f6',
      in_progress: '#f59e0b',
      resolved: '#10b981',
      closed: '#6b7280'
    };
    return colors[status] || '#6b7280';
  };

  const getCategoryLabel = (category: string) => {
    const labels: Record<string, string> = {
      property_inquiry: 'Property Inquiry',
      viewing_request: 'Viewing Request',
      documentation: 'Documentation',
      payment: 'Payment',
      maintenance: 'Maintenance',
      general: 'General'
    };
    return labels[category] || category;
  };

  const getPriorityColor = (priority: string) => {
    const colors: Record<string, string> = {
      low: '#10b981',
      medium: '#f59e0b',
      high: '#ef4444',
      critical: '#dc2626'
    };
    return colors[priority] || '#6b7280';
  };

  return (
    <div style={{ padding: '20px', maxWidth: '1400px', margin: '0 auto' }}>
      <h1 style={{ marginBottom: '20px' }}>Support Ticket Management</h1>
      <p style={{ color: '#6b7280', marginBottom: '20px' }}>Total Tickets: {tickets.length}</p>

      <div style={{ display: 'grid', gridTemplateColumns: '1fr 1.5fr', gap: '20px' }}>
        <div style={{
          backgroundColor: 'white',
          borderRadius: '8px',
          padding: '20px',
          boxShadow: '0 1px 3px rgba(0,0,0,0.1)',
          maxHeight: '80vh',
          overflow: 'auto'
        }}>
          <h2 style={{ marginBottom: '15px' }}>All Tickets</h2>
          {tickets.length === 0 ? (
            <p style={{ color: '#6b7280', textAlign: 'center', padding: '20px' }}>No tickets found</p>
          ) : (
            tickets.map((ticket) => (
              <div
                key={ticket.id}
                onClick={() => viewTicketDetails(ticket.ticket_id)}
                style={{
                  padding: '15px',
                  marginBottom: '10px',
                  borderRadius: '8px',
                  border: selectedTicket?.id === ticket.id ? '2px solid #3b82f6' : '1px solid #e5e7eb',
                  cursor: 'pointer',
                  backgroundColor: selectedTicket?.id === ticket.id ? '#eff6ff' : 'white',
                  transition: 'all 0.2s'
                }}
              >
                <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '5px' }}>
                  <strong style={{ fontSize: '14px' }}>{ticket.ticket_id}</strong>
                  <span style={{
                    padding: '2px 6px',
                    borderRadius: '4px',
                    backgroundColor: getStatusColor(ticket.status) + '20',
                    color: getStatusColor(ticket.status),
                    fontSize: '11px',
                    fontWeight: '500'
                  }}>
                    {ticket.status.replace('_', ' ').toUpperCase()}
                  </span>
                </div>
                <p style={{ fontSize: '13px', marginBottom: '5px', fontWeight: '500' }}>{ticket.subject}</p>
                <div style={{ display: 'flex', justifyContent: 'space-between', fontSize: '12px', color: '#6b7280' }}>
                  <span>{ticket.user_name} ({ticket.user_type})</span>
                  <span>{getCategoryLabel(ticket.category)}</span>
                </div>
                <div style={{ display: 'flex', justifyContent: 'space-between', fontSize: '11px', color: '#9ca3af', marginTop: '5px' }}>
                  <span>{new Date(ticket.created_at).toLocaleDateString()}</span>
                  <span style={{ color: getPriorityColor(ticket.priority) }}>
                    {ticket.priority.toUpperCase()}
                  </span>
                </div>
              </div>
            ))
          )}
        </div>

        <div style={{
          backgroundColor: 'white',
          borderRadius: '8px',
          padding: '20px',
          boxShadow: '0 1px 3px rgba(0,0,0,0.1)',
          maxHeight: '80vh',
          overflow: 'auto'
        }}>
          {!selectedTicket ? (
            <div style={{ textAlign: 'center', padding: '40px', color: '#6b7280' }}>
              <p>Select a ticket to view details</p>
            </div>
          ) : (
            <>
              <div style={{ borderBottom: '2px solid #e5e7eb', paddingBottom: '15px', marginBottom: '20px' }}>
                <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'start', marginBottom: '10px' }}>
                  <div>
                    <h2 style={{ marginBottom: '5px' }}>{selectedTicket.ticket_id}</h2>
                    <p style={{ color: '#6b7280', fontSize: '14px' }}>
                      {selectedTicket.user_name} ({selectedTicket.user_type})
                    </p>
                  </div>
                  <span style={{
                    padding: '6px 12px',
                    borderRadius: '6px',
                    backgroundColor: getStatusColor(selectedTicket.status) + '20',
                    color: getStatusColor(selectedTicket.status),
                    fontSize: '14px',
                    fontWeight: '600'
                  }}>
                    {selectedTicket.status.replace('_', ' ').toUpperCase()}
                  </span>
                </div>

                <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '10px', fontSize: '14px' }}>
                  <div>
                    <span style={{ color: '#6b7280' }}>Category: </span>
                    <strong>{getCategoryLabel(selectedTicket.category)}</strong>
                  </div>
                  <div>
                    <span style={{ color: '#6b7280' }}>Priority: </span>
                    <strong style={{ color: getPriorityColor(selectedTicket.priority) }}>
                      {selectedTicket.priority.toUpperCase()}
                    </strong>
                  </div>
                  {selectedTicket.property_id && (
                    <div>
                      <span style={{ color: '#6b7280' }}>Property ID: </span>
                      <strong>{selectedTicket.property_id}</strong>
                    </div>
                  )}
                  <div>
                    <span style={{ color: '#6b7280' }}>Created: </span>
                    <strong>{new Date(selectedTicket.created_at).toLocaleDateString()}</strong>
                  </div>
                </div>
              </div>

              <div style={{ marginBottom: '20px' }}>
                <p style={{ fontWeight: '600', marginBottom: '10px' }}>Update Status:</p>
                <div style={{ display: 'flex', gap: '10px', flexWrap: 'wrap' }}>
                  <button
                    onClick={() => updateStatus('in_progress')}
                    style={{
                      padding: '8px 16px',
                      backgroundColor: '#f59e0b',
                      color: 'white',
                      border: 'none',
                      borderRadius: '5px',
                      cursor: 'pointer',
                      fontSize: '14px'
                    }}
                  >
                    In Progress
                  </button>
                  <button
                    onClick={() => updateStatus('resolved')}
                    style={{
                      padding: '8px 16px',
                      backgroundColor: '#10b981',
                      color: 'white',
                      border: 'none',
                      borderRadius: '5px',
                      cursor: 'pointer',
                      fontSize: '14px'
                    }}
                  >
                    Resolved
                  </button>
                  <button
                    onClick={() => updateStatus('closed')}
                    style={{
                      padding: '8px 16px',
                      backgroundColor: '#6b7280',
                      color: 'white',
                      border: 'none',
                      borderRadius: '5px',
                      cursor: 'pointer',
                      fontSize: '14px'
                    }}
                  >
                    Closed
                  </button>
                </div>
              </div>

              <div style={{ marginBottom: '20px' }}>
                <h3 style={{ marginBottom: '10px' }}>Subject</h3>
                <p style={{ fontWeight: '500', marginBottom: '10px' }}>{selectedTicket.subject}</p>
                <h3 style={{ marginBottom: '10px' }}>Description</h3>
                <div style={{
                  backgroundColor: '#f9fafb',
                  padding: '15px',
                  borderRadius: '6px',
                  border: '1px solid #e5e7eb'
                }}>
                  {selectedTicket.description}
                </div>
              </div>

              {selectedTicket.attachments.length > 0 && (
                <div style={{ marginBottom: '20px' }}>
                  <h3 style={{ marginBottom: '10px' }}>Attachments</h3>
                  <div style={{ display: 'flex', flexDirection: 'column', gap: '5px' }}>
                    {selectedTicket.attachments.map((att) => (
                      <a
                        key={att.id}
                        href={`${API_URL}/${att.file_path}`}
                        target="_blank"
                        rel="noopener noreferrer"
                        style={{
                          color: '#3b82f6',
                          textDecoration: 'none',
                          fontSize: '14px'
                        }}
                      >
                        📎 {att.file_name}
                      </a>
                    ))}
                  </div>
                </div>
              )}

              <div style={{ marginBottom: '20px' }}>
                <h3 style={{ marginBottom: '10px' }}>Conversation</h3>
                <div style={{ maxHeight: '300px', overflow: 'auto' }}>
                  {selectedTicket.responses.length === 0 ? (
                    <p style={{ color: '#6b7280', fontSize: '14px' }}>No responses yet. Be the first to respond!</p>
                  ) : (
                    selectedTicket.responses.map((response) => (
                      <div
                        key={response.id}
                        style={{
                          backgroundColor: response.responder_type === 'admin' ? '#eff6ff' : '#f9fafb',
                          padding: '12px',
                          borderRadius: '8px',
                          marginBottom: '10px',
                          border: response.responder_type === 'admin' ? '1px solid #bfdbfe' : '1px solid #e5e7eb'
                        }}
                      >
                        <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '5px' }}>
                          <strong style={{ fontSize: '14px' }}>
                            {response.responder_name}
                            <span style={{
                              marginLeft: '8px',
                              padding: '2px 6px',
                              backgroundColor: response.responder_type === 'admin' ? '#3b82f6' : '#6b7280',
                              color: 'white',
                              borderRadius: '4px',
                              fontSize: '11px'
                            }}>
                              {response.responder_type.toUpperCase()}
                            </span>
                          </strong>
                          <span style={{ fontSize: '12px', color: '#6b7280' }}>
                            {new Date(response.created_at).toLocaleString()}
                          </span>
                        </div>
                        <p style={{ fontSize: '14px', margin: 0 }}>{response.message}</p>
                      </div>
                    ))
                  )}
                </div>
              </div>

              <div>
                <h3 style={{ marginBottom: '10px' }}>Send Response</h3>
                <textarea
                  value={responseMessage}
                  onChange={(e) => setResponseMessage(e.target.value)}
                  placeholder="Type your response here..."
                  style={{
                    width: '100%',
                    minHeight: '100px',
                    padding: '10px',
                    borderRadius: '6px',
                    border: '1px solid #d1d5db',
                    fontSize: '14px',
                    marginBottom: '10px',
                    resize: 'vertical'
                  }}
                />
                <button
                  onClick={sendResponse}
                  disabled={loading || !responseMessage.trim()}
                  style={{
                    padding: '10px 20px',
                    backgroundColor: loading || !responseMessage.trim() ? '#9ca3af' : '#3b82f6',
                    color: 'white',
                    border: 'none',
                    borderRadius: '5px',
                    cursor: loading || !responseMessage.trim() ? 'not-allowed' : 'pointer',
                    fontSize: '14px',
                    fontWeight: '500'
                  }}
                >
                  {loading ? 'Sending...' : 'Send Response'}
                </button>
              </div>
            </>
          )}
        </div>
      </div>
    </div>
  );
};

export default AdminSupport;
