import React from "react";
import { useParams } from "react-router-dom";
import {
  TrendingUp,
  FileText,
  Shield,
  BarChart3,
  MapPin,
  Users,
  DollarSign,
  Building,
} from "lucide-react";
import { Button } from "../components/ui/button";
import {
  InvestmentInsights,
  LocalityIntelligence,
  RiskCompliance,
  PropertyDossier,
  NriAssistance,
  BuilderReputation,
  BenchmarkReports,
  LegalHealth,
  CustomReport,
} from "../components/advisory";
import { RentalYieldDebug } from "../components/advisory/RentalYieldDebug";

interface AdvisoryContent {
  title: string;
  description: string;
  features: string[];
  deliverables: string[];
  timeline: string;
  pricing?: string;
  icon: React.ReactNode;
}

const advisoryContent: Record<string, AdvisoryContent> = {
  "investment-insights": {
    title: "Property Investment Insights Report",
    description:
      "Complete financial evaluation combining historical data, current trends, sentiment indicators, and future projections for confident, data-driven investment decisions.",
    features: [
      "Historical appreciation analysis with year-by-year price growth",
      "Future value predictions using AI and demand-supply metrics",
      "ROI projections for 1, 3, 5, and 10-year investment windows",
      "Infrastructure impact score with development project analysis",
      "Buyer sentiment and market activity assessment",
      "Resale potential and liquidity evaluation",
      "Final investment grade with comprehensive scoring",
      "Market positioning and competitive benchmarking",
    ],
    deliverables: [
      "Comprehensive investment analysis report with executive summary",
      "Historical price evolution graphs and trend visualizations",
      "Future value forecasts with best/average/worst case scenarios",
      "ROI breakdown with rental yield and capital appreciation",
      "Infrastructure impact heatmap and development timeline",
      "Market sentiment dashboard with buyer activity metrics",
      "Final investment scorecard with clear Buy/Hold/Avoid recommendation",
    ],
    timeline: "3-5 business days",
    pricing: "Starting from ₹4,999",
    icon: <TrendingUp className="w-8 h-8 text-[#0056D2]" />,
  },
  "locality-intelligence": {
    title: "Locality Intelligence Report",
    description:
      "Complete 360° neighborhood evaluation analyzing safety, infrastructure, convenience, lifestyle, and development potential to understand the true value of the location.",
    features: [
      "Safety & Crime Index with detailed security analysis",
      "Connectivity & Commute Analysis with traffic patterns",
      "Schools, Hospitals & Amenities comprehensive mapping",
      "Pollution, Noise & Greenery environmental assessment",
      "Upcoming Developments & Growth Indicators analysis",
      "Socioeconomic Profile and community demographics",
      "Locality Livability Score with combined ratings",
      "Infrastructure quality and future development timeline",
    ],
    deliverables: [
      "Comprehensive locality scorecard with detailed ratings",
      "Safety analysis report with crime statistics and trends",
      "Connectivity matrix with commute times and transport options",
      "Amenities map with distance and quality ratings",
      "Environmental health report with pollution and greenery data",
      "Development timeline with upcoming infrastructure projects",
      "Socioeconomic profile with demographic insights",
      "Final livability score with investment recommendations",
    ],
    timeline: "2-3 business days",
    pricing: "Starting from ₹2,999",
    icon: <MapPin className="w-8 h-8 text-[#0056D2]" />,
  },
  "risk-compliance": {
    title: "Risk & Compliance Check Report",
    description:
      "Comprehensive legal and regulatory verification ensuring property safety, government compliance, and freedom from disputes or potential legal risks.",
    features: [
      "Government Approval & Compliance Verification with all certificates",
      "Legal Dispute & Litigation Screening for court cases and ownership issues",
      "Title & Ownership Validation with third-party claims verification",
      "Builder/Developer Compliance Track Record and violation history",
      "Risk Score & Compliance Rating with detailed assessment",
      "Buyer Protection Recommendations with legal safeguards",
      "RERA registration and regulatory compliance verification",
      "Environmental clearance and safety standards validation",
    ],
    deliverables: [
      "Comprehensive legal compliance report with government approval status",
      "Litigation screening report with dispute history and resolution status",
      "Title verification certificate with ownership validation",
      "Builder compliance scorecard with track record analysis",
      "Risk assessment matrix with compliance rating and recommendations",
      "Buyer protection checklist with required documents and legal steps",
      "Final compliance certificate with safety recommendation",
    ],
    timeline: "5-7 business days",
    pricing: "Starting from ₹7,999",
    icon: <Shield className="w-8 h-8 text-[#0056D2]" />,
  },
  "property-dossier": {
    title: "Custom Property Dossier",
    description:
      "Personalized, all-in-one property intelligence file consolidating every detail about your selected property into a single, visually rich, easy-to-understand report.",
    features: [
      "Property Overview & Snapshot with complete property intelligence",
      "Technical & Structural Information with construction quality insights",
      "Financial Breakdown with detailed price structure and EMI guidance",
      "Legal & Documentation Checklist with verification status",
      "Connectivity & Locality Insights with commute and safety analysis",
      "Amenities & Lifestyle Score with usage-based ratings",
      "Property Media File with photos, 3D layouts, and drone views",
      "Personalized Notes & Preferences with dynamic updates",
      "Final Expert Recommendation with investment suitability assessment",
    ],
    deliverables: [
      "Complete personalized property dossier with visual intelligence file",
      "Property overview snapshot with livability and growth scores",
      "Technical specifications report with construction quality analysis",
      "Comprehensive financial breakdown with EMI and comparison analysis",
      "Legal documentation checklist with verification status indicators",
      "Connectivity matrix with commute times and locality insights",
      "Amenities scorecard with lifestyle ratings and project comparisons",
      "Curated media collection with photos, 3D layouts, and drone footage",
      "Expert recommendation report with investment suitability and next steps",
    ],
    timeline: "7-10 business days",
    pricing: "Starting from ₹12,999",
    icon: <FileText className="w-8 h-8 text-[#0056D2]" />,
  },
  "nri-assistance": {
    title: "NRI Assistance Desk Report",
    description:
      "Dedicated support for Non-Resident Indians with remote buying, documentation, legal compliance, and end-to-end management of property transactions in India.",
    features: [
      "Regulatory & Compliance Guidance (RBI, FEMA, Income Tax)",
      "POA (Power of Attorney) Support with templates and verification",
      "Virtual Assistance & Remote Evaluation with live tours",
      "End-to-End Documentation & Transaction Support",
      "Negotiation & Deal Handling with expert-led support",
      "Post-Purchase Support with rental and maintenance management",
      "Secure Communication Hub with 24x7 NRI-exclusive support",
      "Cross-timezone assistance with global accessibility",
    ],
    deliverables: [
      "Complete NRI compliance guide with RBI, FEMA, and tax regulations",
      "POA documentation package with templates and verification tracker",
      "Virtual property evaluation report with drone views and live tours",
      "End-to-end transaction support with digital documentation assistance",
      "Negotiation support report with pricing analysis and deal optimization",
      "Post-purchase management plan with rental and maintenance services",
      "Secure communication portal with dedicated NRI support channel",
    ],
    timeline: "3-5 business days",
    pricing: "Starting from ₹5,999",
    icon: <Users className="w-8 h-8 text-[#0056D2]" />,
  },
  "builder-reputation": {
    title: "Builder Reputation & Track Record Report",
    description:
      "Comprehensive evaluation of builder credibility, performance, and reliability using verified data, RERA records, customer ratings, and financial stability analysis.",
    features: [
      "Builder Delivery Track Record with on-time vs delayed project statistics",
      "Construction Quality Score with materials audit and durability evaluation",
      "Customer Satisfaction & Review Analytics with aggregated ratings",
      "RERA Compliance & Litigations with legal disputes investigation",
      "Financial Stability Check with market presence and credit analysis",
      "Builder Brand Value & Market Standing with awards and rankings",
      "Transparency & Documentation Score with ethical practices evaluation",
      "Visual Insights & Analytics with timeline charts and comparison data",
    ],
    deliverables: [
      "Complete builder profile with delivery track record and delay statistics",
      "Construction quality assessment with materials audit and durability scores",
      "Customer satisfaction report with review analytics and complaint ratios",
      "RERA compliance verification with legal disputes and penalty history",
      "Financial stability analysis with market presence and credit evaluation",
      "Brand value assessment with awards, rankings, and market reputation",
      "Transparency scorecard with documentation and ethical practices review",
      "Visual analytics dashboard with timeline charts and builder comparisons",
    ],
    timeline: "7-10 business days",
    pricing: "Starting from ₹8,999",
    icon: <Building className="w-8 h-8 text-[#0056D2]" />,
  },
  "benchmark-reports": {
    title: "Comparative Property Benchmark Report",
    description:
      "Side-by-side comparison of multiple properties using pricing, ROI, amenities, builder score, construction quality, locality growth, and long-term value analysis.",
    features: [
      "Side-by-Side Property Comparison with price per sq ft and total costs",
      "Amenity Benchmarking with quality ratings and unique features analysis",
      "Construction & Age Comparison with quality and safety scores",
      "Locality Performance Comparison with growth and connectivity metrics",
      "Builder Reputation Comparison with track record and delivery consistency",
      "ROI & Rental Yield Benchmark with appreciation and value analysis",
      "Visual Insights with comparison charts and ranking systems",
      "Value-for-money index with comprehensive scoring methodology",
    ],
    deliverables: [
      "Side-by-side property comparison matrix with detailed cost analysis",
      "Amenity benchmarking report with quality ratings and feature scoring",
      "Construction quality comparison with age and safety assessments",
      "Locality performance analysis with growth indicators and connectivity scores",
      "Builder reputation comparison with track record and consistency metrics",
      "ROI and rental yield benchmark with appreciation trends and forecasts",
      "Visual comparison dashboard with charts, rankings, and value meters",
      "Final recommendation report with best value property identification",
    ],
    timeline: "3-5 business days",
    pricing: "Starting from ₹4,999",
    icon: <BarChart3 className="w-8 h-8 text-[#0056D2]" />,
  },
  "legal-health": {
    title: "Legal Health & Title Status Report",
    description:
      "Comprehensive legal safety verification through ownership verification, documentation review, encumbrance checks, and land-use compliance analysis.",
    features: [
      "Ownership Verification with current and past owners tracking",
      "Document Validation including sale deed, khata, and EC analysis",
      "Litigation & Dispute Check with court cases and RERA complaints",
      "Encumbrance & Liability assessment with loans and mortgage check",
      "Land Use & Zoning Compliance with BBMP/BDA approvals",
      "Project Approval Status with RERA and building licenses",
      "Legal document verification with status badges and checklists",
      "Complete legal safety certification for secure investment",
    ],
    deliverables: [
      "Ownership verification report with title clarity score and tracking",
      "Document validation certificate with sale deed and khata verification",
      "Litigation screening report with court cases and dispute history",
      "Encumbrance analysis with loans, mortgage, and liability assessment",
      "Land use compliance report with zoning and master plan alignment",
      "Project approval verification with RERA, OC, and CC status",
      "Legal health scorecard with verification seals and status indicators",
      "Final legal safety certification with risk assessment and clearance",
    ],
    timeline: "7-10 business days",
    pricing: "Starting from ₹9,999",
    icon: <Shield className="w-8 h-8 text-[#0056D2]" />,
  },
  "rental-yield": {
    title: "Rental Yield Analysis Report",
    description:
      "Comprehensive rental profitability analysis using yield calculations, demand trends, and locality-based rental analytics for informed investment decisions.",
    features: [
      "Expected Monthly Rent Analysis with market rates and seasonal variations",
      "Rental Demand Score with heatmap and tenant profile analysis",
      "Occupancy Trends with peak seasons and vacancy risk assessment",
      "Net Rental Yield Calculation with maintenance, tax, and fee deductions",
      "Locality Comparison with nearby areas and rent growth analysis",
      "Break-Even & Return Forecast with investment recovery timeline",
      "Visual Insights with rent graphs, yield calculators, and heatmaps",
      "Profitability assessment with cash flow forecasting and ROI analysis",
    ],
    deliverables: [
      "Monthly rent analysis report with furnished vs unfurnished comparisons",
      "Rental demand assessment with tenant profiles and market heatmap",
      "Occupancy trends analysis with seasonal patterns and vacancy risks",
      "Net rental yield calculation with comprehensive cost deductions",
      "Locality benchmarking report with 5-year rent growth projections",
      "Break-even analysis with investment recovery timeline and forecasts",
      "Visual rental intelligence dashboard with graphs and profitability meters",
      "Final investment recommendation with rental income optimization strategies",
    ],
    timeline: "3-5 business days",
    pricing: "Starting from ₹3,999",
    icon: <DollarSign className="w-8 h-8 text-[#0056D2]" />,
  },
  "custom-report": {
    title: "Custom Advisory Report",
    description:
      "Personalized consultation tool providing custom insights based on user preferences, budget, goals, and locality interest with comprehensive risk evaluation and market analysis.",
    features: [
      "Personalized Property Recommendations based on budget and purpose",
      "Buyer's Risk Score Evaluation across financial, legal, builder, and market risks",
      "Best Time to Buy Analysis with seasonal trends and price forecasts",
      "Negotiation Guidance with ideal margins and market comparisons",
      "Financial Planning with EMI calculators and tax benefits",
      "Custom Property Comparisons with detailed pros & cons analysis",
      "Visual Insights with advisor icons and decision flowcharts",
      "Investment window optimization and demand consistency analysis",
    ],
    deliverables: [
      "Personalized property recommendations matching your profile",
      "Comprehensive risk assessment with score meters for all risk categories",
      "Market timing analysis with seasonal trends and price forecasts",
      "Negotiation strategy guide with builder offers and discount insights",
      "Complete financial planning toolkit with loan comparisons",
      "Side-by-side property comparison with value-for-money rankings",
      "Visual decision support with flowcharts and advisor recommendations",
    ],
    timeline: "7-10 business days",
    pricing: "Starting from ₹12,999",
    icon: <FileText className="w-8 h-8 text-[#0056D2]" />,
  },
};

export function AdvisoryPage() {
  const { advisoryId } = useParams<{ advisoryId: string }>();
  const [selectedCity, setSelectedCity] = React.useState('Bangalore');
  const advisory = advisoryContent[advisoryId || ""];

  if (!advisory) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-4">
            Advisory Service Not Found
          </h1>
          <p className="text-gray-600">
            The requested advisory service could not be found.
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="bg-[#0056D2] text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <div className="flex justify-center mb-4">{advisory.icon}</div>
            <h1 className="text-3xl md:text-4xl font-bold mb-4">
              {advisory.title}
            </h1>
            <p className="text-lg md:text-xl text-blue-100 max-w-4xl mx-auto leading-relaxed">
              {advisory.description}
            </p>
          </div>
        </div>
      </div>

      {/* Detailed Sections for each advisory type */}
      {advisoryId === "investment-insights" && <InvestmentInsights />}
      {advisoryId === "locality-intelligence" && <LocalityIntelligence />}
      {advisoryId === "risk-compliance" && <RiskCompliance />}
      {advisoryId === "property-dossier" && <PropertyDossier />}
      {advisoryId === "nri-assistance" && <NriAssistance />}
      {advisoryId === "builder-reputation" && <BuilderReputation />}
      {advisoryId === "benchmark-reports" && <BenchmarkReports />}
      {advisoryId === "legal-health" && <LegalHealth />}
      {advisoryId === "rental-yield" && <RentalYieldDebug />}
      {advisoryId === "custom-report" && <CustomReport />}

      {/* Features & Deliverables Section for other advisory types */}
      {!["investment-insights", "locality-intelligence", "risk-compliance", "property-dossier", "nri-assistance", "builder-reputation", "benchmark-reports", "legal-health", "rental-yield", "custom-report"].includes(advisoryId || "") && (
        <div className="bg-white py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
              {/* Features */}
              <div>
                <h2 className="text-2xl font-bold text-gray-900 mb-6">
                  Key Features
                </h2>
                <ul className="space-y-4">
                  {advisory.features.map((feature, index) => (
                    <li key={index} className="flex items-start gap-3">
                      <div className="w-2 h-2 bg-[#0056D2] rounded-full mt-2 flex-shrink-0"></div>
                      <span className="text-gray-700">{feature}</span>
                    </li>
                  ))}
                </ul>
              </div>

              {/* Deliverables */}
              <div>
                <h2 className="text-2xl font-bold text-gray-900 mb-6">
                  What You'll Receive
                </h2>
                <ul className="space-y-4">
                  {advisory.deliverables.map((deliverable, index) => (
                    <li key={index} className="flex items-start gap-3">
                      <div className="w-2 h-2 bg-green-500 rounded-full mt-2 flex-shrink-0"></div>
                      <span className="text-gray-700">{deliverable}</span>
                    </li>
                  ))}
                </ul>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* CTA Section */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Ready to Get Started?
          </h2>
          <p className="text-lg text-gray-600 mb-8">
            Get your comprehensive {advisory.title.toLowerCase()} delivered in {advisory.timeline}.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
            <div className="text-2xl font-bold text-[#0056D2]">
              {advisory.pricing}
            </div>
            <Button size="lg" className="bg-[#0056D2] hover:bg-[#003d9a]">
              Order Report Now
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
}
